﻿using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using GuitarTrainer.GuitarProComponents;

namespace GuitarTrainer.GUI
{
    /// <summary>
    /// Displays notes in a guitar tablature representation
    /// </summary>
    class BarTablature : BarMtp
    {
        // How much space on the x-axis to the right of the note is for effects on note
        private const int EONX = 5;

        // Effects on notes labels
        private readonly List<Label> eonLabels;

        // Silence labels
        private readonly List<Label> sLabels;

        // Default constructor
        public BarTablature(DisplayOptions dispOps, BarPanel bp)
            : base(dispOps, bp)
        {
            eonLabels = new List<Label>();
            sLabels = new List<Label>();
        }

        /// <summary>
        /// Add a silence symbol using the currentX position and duration
        /// </summary>
        /// <param name="currentX">The current x position</param>
        /// <param name="d">the duration of the silence</param>
        private void addSilence(int currentX, GPDuration d)
        {
            Label silenceLabel = new Label {Image = Form1.silences[d.getIndex()]};

            locateSilence(silenceLabel, currentX);
            silenceLabel.Visible = true;

            Controls.Add(silenceLabel);
            sLabels.Add(silenceLabel);
        }

        /// <summary>
        /// Calculates the height of a string
        /// </summary>
        /// <param name="s">The line number of which the string is being written to</param>
        /// <returns></returns>
        protected override int calculateHeightOfString(int s)
        {
            // The higher the value of the string, the closer it is to the top
            int y = DispOptions.TopOffset + BPanel.H - s * DispOptions.LineSpacing;

            // Strings are represented in a 7-string scale
            // The line below locates the notes according to that
            y += (7 - BPanel.Lines) * DispOptions.LineSpacing;

            // Center the text in the middle, vertically
            y += FontAscent / 2;

            return y;
        }

        /// <summary>
        /// Creates a label for some effect (slide, hammer-on, etc)
        /// </summary>
        /// <param name="pos">The position of the effect</param>
        /// <param name="x">The x coordinate of the effect</param>
        /// <param name="y">The y coordinate of the effect</param>
        /// <param name="eon">The effect on note to draw</param>
        /// <param name="beat">The beat to draw the effect on</param>
        /// <returns>The label created for this particular effect</returns>
        private Label createLabelForIndex(int pos, int x, int y, GPEffectsOnNote eon, GPBeat beat)
        {
            SlideLabel sl;
            Label toReturn = null;

            int index = -1 * pos;

            // If the index indicates that it is a slide label
            if ((index >= 1) && (index <= 6))
            {
                int width = calculateWidthOfBeat(beat);
                sl = new SlideLabel(DispOptions, width/*, panel*/);
                sl.setTypeOfSlide(eon.slide);
                sl.setAnchor(x, y);
                toReturn = sl;
            }

            return toReturn;
        }

        /// <summary>
        /// Add the effect on note with the current position and duration
        /// </summary>
        /// <param name="currentX">The current x coordinate of the effect</param>
        /// <param name="currentY">The current y coordinate of the effect</param>
        /// <param name="eon">The effect to draw</param>
        /// <param name="beat">The beat the effect is being drawn on</param>
        private void eonAdd(int currentX, int currentY, GPEffectsOnNote eon, GPBeat beat)
        {
            Label eonLabel;
            int pos = index(eon);

            // If it is a GIF effect on note
            if (pos >= 0)
            {
                eonLabel = new Label {Image = Form1.eons[pos]};

                eonLocate(eonLabel, currentX, currentX, eon, beat);
            }
            else
            {
                // Create label and sets the location of the slides
                eonLabel = createLabelForIndex(pos, currentX, currentY, eon, beat);
            }

            if (pos == 0)
            {
                eonLabel.Text = "?";
                eonLabel.Name = "?";
                
                eonLabel.Visible = DispOptions.DisplayUnsupportedEffects;
            }
            else
            {
                eonLabel.Text = pos.ToString();
                eonLabel.Name = pos.ToString();

                eonLabel.Visible = true;
            }

            // Set the tool tip next to the string representation of the effects on note
            eonLabels.Add(eonLabel);
        }

        /// <summary>
        /// Sets the position of an effects on note label
        /// </summary>
        /// <param name="eonLabel">The label to set the position of</param>
        /// <param name="currentX">The current x value</param>
        /// <param name="currentY">The current y value</param>
        /// <param name="eon">The effect on note being drawn</param>
        /// <param name="beat">The beat the eon is being drawn for</param>
        private void eonLocate(Label eonLabel, int currentX, int currentY, GPEffectsOnNote eon, GPBeat beat)
        {
            Point p2;

            Size s = eonLabel.PreferredSize;

            Point p = new Point(currentX, currentY);
            SlideLabel sl = new SlideLabel();

            // If it is a slide label
            if (sl.GetType().IsInstanceOfType(eonLabel))
            {
                sl = (SlideLabel)eonLabel;

                sl.setAnchor(p);
                sl.Width = calculateWidthOfBeat(beat);
            }
            // It's a GIF label
            else
            {
                p2 = eonPosition(eon, p, s);
                eonLabel.SetBounds(p2.X, p2.Y, s.Width, s.Height);
            }
        }

        /// <summary>
        /// Calculates the effects position
        /// </summary>
        /// <param name="eon">The effect to calculate the position of</param>
        /// <param name="current">The current point of the effect</param>
        /// <param name="s">The size of the effect</param>
        /// <returns>The location of the effect</returns>
        private Point eonPosition(GPEffectsOnNote eon, Point current, Size s)
        {
            int whichEon = index(eon);
            Point toReturn = current;

            // "Let-ring" is displayed
            if (whichEon == 1)
            {
                toReturn.X = current.X + EONX;
                toReturn.Y = current.Y - s.Height / 2;
            }

            // Unsupported effects have an index of 0 and are located at the same
            // position as bends
            else if ((whichEon == 0) || ((whichEon >= 2) && (whichEon <= 6)))
            {
                toReturn.X = current.X + EONX;
                toReturn.Y = current.Y - 3 * s.Height / 2 + 2;
            }

            return toReturn;
        }

        /// <summary>
        /// Determines whether or not an object is equal to this
        /// </summary>
        /// <param name="obj">The object being tested</param>
        /// <returns>Whether or not the object is equal to this</returns>
        public override bool Equals(object obj)
        {
            BarTablature other;
            bool toReturn = false;

            if ((obj != null) && (obj.GetType().IsInstanceOfType(this)))
            {
                other = (BarTablature)obj;
                toReturn = MeasureTrackPair.Equals(other.MeasureTrackPair);
            }

            return toReturn;
        }

        /// <summary>
        /// Returns the index of a bend type
        /// </summary>
        /// <param name="bendType">The bend type to return the index of</param>
        /// <returns>The index of the bend type</returns>
        private static int index(GPBendType bendType)
        {
            // the hard coded values of i are defined by the GuitarPro file format
            int i = -1;

            if (bendType != null)
            {
                if (bendType.Equals(GPBendType.BEND))
                    i = 0;
                else if (bendType.Equals(GPBendType.BEND_RELEASE))
                    i = 1;
                else if (bendType.Equals(GPBendType.BEND_RELEASE_BEND))
                    i = 2;
                else if (bendType.Equals(GPBendType.PREBEND))
                    i = 3;
                else if (bendType.Equals(GPBendType.PREBEND_RELEASE))
                    i = 4;
            }

            return i;
        }

        /// <summary>
        /// Returns the index of an effects on note
        /// </summary>
        /// <param name="eon">The effect to return the index of</param>
        /// <returns>The index of the effect</returns>
        private static int index(GPEffectsOnNote eon)
        {
            // the hard coded values of i are defined by the GuitarPro file format
            int i = -1;

            if (eon != null)
            {
                i = 0;
                if (eon.letRing)
                    i = 1;
                else if (eon.bend != null)
                    i = 2 + index(eon.bend.Type);
                else if (eon.slide != null)
                    i = -1 * (1 + eon.slide.getIndex());
            }

            return i;
        }

        /// <summary>
        /// Sets the position of the silence
        /// </summary>
        /// <param name="silenceLabel">The silence label to set the index of</param>
        /// <param name="currentX">The current x coordinate of the label</param>
        private void locateSilence(Control silenceLabel, int currentX)
        {
            Size s = silenceLabel.PreferredSize;

            int y = DispOptions.TopOffset + BPanel.H / 2 - s.Height / 2;
            int x = currentX - s.Width / 2;

            silenceLabel.SetBounds(x, y, s.Width, s.Height);
        }

        /// <summary>
        /// Fill a rectangle with the background color of the size of the text to be displayed
        /// </summary>
        /// <param name="g">The graphics object to draw the space around fret with</param>
        /// <param name="s">The string being drawn (like the number "3")</param>
        /// <param name="x">The x coordinate of the string being drawn</param>
        /// <param name="y">The y coordinate of the string being drawn</param>
        /// <returns>The new x coordinate of the string to be drawn</returns>
        private int drawSpaceAroundFret(Graphics g, string s, int x, int y)
        {
            Color c = BackColor;

            // Calc the width of the text
            int dx = (int)g.MeasureString(s, BarPanel.notesFont).Width;

            // Center the text
            int newX = x - dx / 2;

            g.FillRectangle(new SolidBrush(c), newX, y - FontAscent, dx + 2, FontAscent + 1);

            return newX;
        }

        /// <summary>
        /// Draws and effects on the bar tablature
        /// </summary>
        /// <param name="currX">The current x coordinate of the effect</param>
        /// <param name="y">The current y coordinate of the effect</param>
        /// <param name="eon">The effect to be drawn</param>
        /// <param name="countEON">The length of the effect</param>
        /// <param name="beat">The beat the effect is being drawn on</param>
        protected override void drawEffects(int currX, int y, GPEffectsOnNote eon, int countEON, GPBeat beat)
        {
            Label eonLabel;

            if (eon == null) return;

            if (AddEONs)
                eonAdd(currX, y, eon, beat);
            
            // The next time labels are relocated
            else
            {
                eonLabel = eonLabels[countEON];

                // If the eon label is an unsupported effect, draw a "?"
                if (eonLabel.Name.Equals("?"))
                    eonLabel.Visible = DispOptions.DisplayUnsupportedEffects;

                eonLocate(eonLabel, currX, y, eon, beat);
            }
        }

        /// <summary>
        /// Draw a note in this bar tablature
        /// </summary>
        /// <param name="g">The graphics object to be used to draw the note</param>
        /// <param name="currentX">The current x coordinate of the note</param>
        /// <param name="y">The current y coordinate of the note</param>
        /// <param name="note">The note to be drawn</param>
        /// <param name="noteColor">The color the note should be drawn with</param>
        protected override void drawNote(Graphics g, int currentX, int y, GPNote note, Color noteColor)
        {
            int pos;
            GPDynamic dynamic;

            int fret = note.FretNumber;
            string s = fret.ToString();

            // Draw the background space around the displayed fret
            int x = drawSpaceAroundFret(g, s, currentX, y);

            if (DispOptions.FretColoringType == DisplayOptions.RELATED_TO_DURATION)
            {
                // Set the color of the fret according to the note duration
                if (note.Duration != null)
                {
                    pos = note.Duration.getIndex();
                    noteColor = DispOptions.FretColors.getColor(pos);
                }
            }
            else if (DispOptions.FretColoringType == DisplayOptions.RELATED_TO_DYNAMIC)
            {
                dynamic = note.Dynamic;
                if (dynamic != null)
                {
                    // Since index 0 is invalid, the first valid dynamic is 1, but 
                    // pos is used to access the array
                    pos = dynamic.getIndex() - 1;

                    noteColor = DispOptions.FretColors.getColor(pos);
                }
            }
            Point p = new Point(x + 1, y - 6);
            g.DrawString(s, BarPanel.notesFont, new SolidBrush(noteColor), p);
        }

        /// <summary>
        /// Draw a silence on this bar tablature
        /// </summary>
        /// <param name="currX">The current x coordinate of the silence</param>
        /// <param name="beat">The current beat of the silence</param>
        /// <param name="contSilences">The length of the silence</param>
        protected override void drawSilence(int currX, GPBeat beat, int contSilences)
        {
            if (AddSilence)
                addSilence(currX, beat.Duration);
            else
                locateSilence(sLabels[contSilences], currX);
        }
    }
}
