﻿namespace GuitarTrainer.GuitarProComponents
{
    /// <summary>
    /// Describes the chord diagram for this piece
    /// </summary>
    class GPChordDiagram
    {
        /**
         * Frets: List of 7 integers
         * Corresponds to the fret number played on each string, from the highest to the lowest
         * -1 indicates a string unplayed (or muted)
         * 0 indcates an open string (not fretted, but played)
         */
        private readonly int[] frets;

        /**
         * Fret of the barre: List of 5 bytes
         * Indicates the fret number of each possible barre
         */
        private readonly int[] fretOfBarres;

        /**
         * Barre start: List of 5 bytes
         * Indicates the first string of the barre, 1 being the highest (high E string, on guitar)
         * The list order matches the chord different barres frets list order
         */
        private readonly int[] barreStarts;

        /**
         * Barre end: List of 5 bytes
         * Indicates the first string of the barre, 1 being the lowest
         * The list order matches the chord different barres frest list order
         */
        private readonly int[] barreEnds;

        /**
         * Fingering: List of 7 bytes
         * Describes the fingering used to play the chord
         * Below is given the fingering from the highest string to lowest:
         * -2: unknown finger
         * -1: X or 0 (muted or open, ie no finger used)
         * 0: thumb
         * 1: index
         * 2: middle
         * 3: ring
         * 4: pinkey
         */
        private readonly GPFingering[] fingering;

        /// <summary>
        /// Default constructor
        /// </summary>
        public GPChordDiagram()
        {
            Sharp = false;
            Root = null;
            ChordType = null;
            NineElevenThirteen = 0;
            Bass = null;
            TonalityType = null;
            AddedNote = 0;
            Name = "";
            TonalityFive = null;
            TonalityNine = null;
            TonalityEleven = null;
            BaseFret = 0;
            frets = new int[7];
            NumBarres = 0;
            fretOfBarres = new int[5];
            barreStarts = new int[5];
            barreEnds = new int[5];
            fingering = new GPFingering[7];
            ChordFingeringDisplayed = false;
        }

        /// <summary>
        /// Returns a string representation of the chord diagram
        /// </summary>
        /// <returns>The string representation of this class</returns>
        public string toString()
        {
            string toReturn = "isSharp: " + Sharp;
            if(Root != null)
                toReturn += "\n\t\tRoot: " + Root.toString();
            if(ChordType != null)
                toReturn += "\n\t\tChordType: " + ChordType.toString();
            if(Bass != null)
                toReturn += "\n\t\tBass: " + Bass.toString();
            if(TonalityType != null)
                toReturn += "\n\t\tTonality: " + TonalityType.ToString();
            toReturn += "\n\t\tAddedNote: " + AddedNote;
            toReturn += "\\n\t\tName: " + Name;
            if(TonalityFive != null)
                toReturn += "\n\t\tTonality Five: " + TonalityFive.ToString();
            if(TonalityNine != null)
                toReturn += "\n\t\tTonality Nine: " + TonalityNine.ToString();
            if(TonalityEleven != null)
                toReturn += "\n\t\tTonality Eleven: " + TonalityEleven.ToString();
            toReturn += "\n\t\tBase Fret: " + BaseFret;
            toReturn += "\n\t\tFrets from left to right (including the 7th guitar string): ";
            for(int i = 6; i >= 0; i--)
            {
                toReturn += frets[i];
                if(i - 1 >= 0)
                    toReturn += ", ";
            }

            toReturn += "\n\t\tIsChordFingeringDisplayed: " + ChordFingeringDisplayed;
            if(fingering != null)
            {
                toReturn += "\n\t\tFinger: ";
                for(int i = 6; (i >= 0) && (fingering[i] != null); i--)
                {
                    toReturn += fingering[i].toString();
                    if(i-1 >= 0)
                        toReturn += ", ";
                }
            }

            toReturn += "\n\t\tNum Barres: " + NumBarres;

            toReturn += "\n\t\tFrets of Barres: ";
            for(int i = 0; i < 5; i++)
            {
                toReturn += fretOfBarres[i];
                if(i+1 < 5)
                    toReturn += ", ";
            }

            toReturn += "\n\t\tBarres Starts: ";
            for(int i = 0; i < 5; i++)
            {
                toReturn += barreStarts[i];
                if(i+1 < 5)
                    toReturn += ", ";
            }

            toReturn += "\n\t\tBarres Ends: ";
            for(int i = 0; i < 5; i++)
            {
                toReturn += barreEnds[i];
                if(i+1 < 5)
                    toReturn += ", ";
            }

            return toReturn;
        }

        #region Getters and setters

        // Determines if the chord is displayed as sharp or flat
        public bool Sharp { private get; set; }

        // The root of the diagram (-1 for the custimzed chords, 0 when root is C, 
        // 1 = C#, etc
        public GPChordNote Root { private get; set; }

        /**
         * Determines the chord type as followed:
         * 0: M
         * 1: 7
         * 2: 7M
         * 3: 6
         * 4: m
         * 5: m7
         * 6: m7M
         * 7: m6
         * 8: sus2
         * 9: sus4
         * 10: 7sus2
         * 11: 7sus4
         * 12: dim
         * 13: aug
         * 14: 5
         */
        public GPChordType ChordType { private get; set; }

        // Nine, Eleven of Thirteen: byte
        // Determines if the chord goes until the ninth, the eleventh, or the thirteenth
        public int NineElevenThirteen { private get; set; }

        // Bass: int
        // Lowest note of the chord. It gives the chord inversions
        public GPChordNote Bass { get; set; }

        /**
         * Dimished/Augmented int
         * Tonality linked with 9/11/13
         * 0: perfect ("juste")
         * 1: augmented
         * 2: dimished
         */
        public GPTonalityType TonalityType { get; set; }

        // add: byte
        // Allows determination if an 'add' (added note) is present in the chord
        public int AddedNote { get; set; }

        // Name: string
        // 20 character long string containing the chord name
        public string Name { get; set; }

        /**
         * Fifth: byte
         * Tonality of the fifth:
         * 0: perfect ("juste")'
         * 1: augmented
         * 2: diminished
         */
        public GPTonalityType TonalityFive { get; set; }

        /**
         * Ninth: byte
         * Tonality of the ninth:
         * 0: perfect ("juste")
         * 1: augmented
         * 2: dimished
         * This tonality is valid only if the value "Nine, Eleven, or Thirteen" is 11 or 13
         */
        public GPTonalityType TonalityNine { get; set; }

        /**
         * Eleventh: byte
         * Tonality of the eleventh:
         * 0: perfect ("juste")
         * 1: augmented
         * 2: dimished
         * This tonality is valid only if the value "Nine, Eleven, or Thirteen" is 13
         */
        public GPTonalityType TonalityEleven { get; set; }

        // Base fret: integer
        // Shows the fret from which the chord is displayed
        public int BaseFret { get; set; }

        public void setFret(int pos, int fret)
        {
            if ((pos >= 1) && (pos <= 7))
                frets[pos - 1] = fret;
        }

        /**
         * Number of barres: byte
         * Indicates the number of barres there are in the chord
         * A chord can have up to 5 barres.
         */
        public int NumBarres { get; set; }

        public void setFretOfBarre(int pos, int fret)
        {
            if ((pos >= 1) && (pos <= 5))
                fretOfBarres[pos - 1] = fret;
        }
        public void setBarreStart(int pos, int barre)
        {
            if ((pos >= 1) && (pos <= 5))
                barreStarts[pos - 1] = barre;
        }
        public void setBarreEnd(int pos, int barre)
        {
            if ((pos >= 1) && (pos <= 5))
                barreEnds[pos - 1] = barre;
        }
        public void setFingering(int pos, int finger)
        {
            if ((pos >= 1) && (pos <= 7))
                fingering[pos - 1] = GPFingering.valueOf(finger);
        }

        /**
         * ShowDiagFinger: byte
         * If it is 0x01, the chord fingering is displayed
         * If it is 0x00, the chord fingering is hidden
         */
        public bool ChordFingeringDisplayed { get; set; }

        #endregion
    }
}
