﻿using System;

namespace GuitarTrainer.GuitarProComponents
{
    /// <summary>
    /// The class that describes a duration read from a Guitar Pro file.
    /// Note, however, that there is still an undocumented (by Guitar Pro) 
    /// duration used by this class
    /// </summary>
    class GPDuration
    {
        // A duration is represented by an integer
        private readonly int type;

        // Different types of note lengths that this class accepts
        public static readonly GPDuration WHOLE = new GPDuration(-2);
        public static readonly GPDuration HALF = new GPDuration(-1);
        public static readonly GPDuration QUARTER = new GPDuration(0);
        public static readonly GPDuration EIGHTH = new GPDuration(1);
        public static readonly GPDuration SIXTEENTH = new GPDuration(2);
        public static readonly GPDuration THIRTY_SECOND = new GPDuration(3);
        public static readonly GPDuration SIXTY_FOURTH = new GPDuration(4);
        public static readonly GPDuration HUNDRED_TWENTY_EIGHTH = new GPDuration(5);
        public static readonly GPDuration UNKNOWN_DURATION = new GPDuration(6);

        // The durations
        private static readonly GPDuration[] durations = { WHOLE, HALF, QUARTER, EIGHTH,
            SIXTEENTH, THIRTY_SECOND, SIXTY_FOURTH, HUNDRED_TWENTY_EIGHTH,
            UNKNOWN_DURATION 
        };

        // The durations in string format
        private static readonly string[] durationNames = { "Whole note", "Half note",
            "Quarter note", "Eighth note", "Sixteenth note",
            "Thirty-second note", "Sixty-fourth note",
            "Hundred twenty-eighth note", "UNDOCUMENTED DURATION, VALUE = 6" 
        };

        // Creates a new duration whose type is the argument 'type'
        private GPDuration(int type)
        {
            this.type = type;
        }

        /// <summary>
        /// Compares this duration to another object. Returns the difference in the two
        /// types of durations.
        /// </summary>
        /// <param name="o">The duration we are comparing</param>
        /// <returns>The difference of this object with the argument. 0 means a match.</returns>
        public int compareTo(Object o)
        {
            if (o is GPDuration)
                return ((GPDuration)o).type - type;
            
            return -1;
        }

        // Returns true if the supplied argument object is equal to this
        public override bool Equals(Object o)
        {
            if (o is GPDuration)
                return type == ((GPDuration)o).type;
            
            return false;
        }

        /// <summary>
        /// Returns this duration's index
        /// </summary>
        /// <returns>The index value of this duration</returns>
        public int getIndex()
        {
            return (type + 2);
        }

        /// <summary>
        /// Returns the duration assocated with thie value of the argument 'type'
        /// </summary>
        /// <param name="type">The type to return the duration of</param>
        /// <returns>The duration that the type represents</returns>
        public static GPDuration valueOf(int type)
        {
            return durations[type + 2];
        }

        /// <summary>
        /// Returns a string representation of this duration
        /// </summary>
        /// <returns>The string representation of this</returns>
        public override string ToString()
        {
            return durationNames[type + 2];
        }
    }
}
