﻿using System.Collections.Generic;
using System.Linq;
using System.IO;

namespace GuitarTrainer.GuitarProComponents
{
    /// <summary>
    /// Class used to join different GP4Piece object together
    /// </summary>
    class GPSong
    {
        // Triplet Feel
        public bool tripletFeel;

        // Title of the piece
        private string title;

        // Subtitle of the piece
        private string subtitle;

        // Interpret of the piece
        private string interpret;

        // Album of the piece
        private string album;

        // Octave of the piece
        private int octave;

        // Author of the song
        private string authorSong;

        // Copyright holder of the piece
        private string copyright;

        // Author of this tablature
        private string authorPiece;

        // Instructions to be used with this piece
        private string instructions;

        // Notes about this piece
        private string note;

        // Lyrics associated with the song
        private GPTrackLyrics lyrics;

        // Tempo of the piece
        private int tempo;

        // The MIDI channels used
        private readonly GPMIDIChannel[] channels;

        // List of tracks present in this song
        public readonly List<GPTrack> tracks;

        // List of measures present in the song
        public readonly List<GPMeasure> measures;

        // List of the measures/tracks pairs contain info about the music
        public readonly List<GPMeasureTrackPair> measuresTracksPairs;

        // The version of the file the piece was loaded from
        private string version;

        // List of chord diagrams
        private readonly List<GPChordDiagram> chordsDiagrams;

        // The key signature at the beginning of the piece
        private GPKey key;

        // Constructor with an "empty" song
        public GPSong()
        {
            album = "";
            authorPiece = "";
            channels = new GPMIDIChannel[64];
            chordsDiagrams = new List<GPChordDiagram>();
            copyright = "";
            instructions = "";
            interpret = "";
            key = GPKey.g;
            lyrics = new GPTrackLyrics();
            measures = new List<GPMeasure>();
            measuresTracksPairs = new List<GPMeasureTrackPair>();
            note = "";
            octave = 8;
            subtitle = "";
            tempo = 100;
            title = "";
            tracks = new List<GPTrack>();
            tripletFeel = false;
            version = "unknown";
        }

        /// <summary>
        /// Prints out the ISong at some depth
        /// </summary>
        /// <param name="deep">How deep into the song to print out information</param>
        /// <returns>The string representation of this class to some depth</returns>
        public string ToString(int deep)
        {
            int tabsStringImpl = 0;

            StreamWriter sr = new StreamWriter("output.txt");

            sr.Write("GP - BEGIN\n");

            // Print the header stats
            tabsStringImpl++;
            sr.Write("HEADER - BEGIN\n");
            sr.Write("Version: " + version + "\n");
            sr.Write("Title: " + title + "\n");
            sr.Write("Subtitle: " + subtitle + "\n");
            sr.Write("Interpret: " + interpret + "\n");
            sr.Write("Album: " + album + "\n");
            sr.Write("Author of the song: " + authorSong + "\n");
            sr.Write("Copyright: " + copyright + "\n");
            sr.Write("Author of the piece: " + authorPiece + "\n");
            sr.Write("Instructions: " + instructions + "\n");
            sr.Write("Notes: " + note + "\n");
            sr.Write("Use triplet feel? " + tripletFeel + "\n");
            sr.Write(lyrics.toString());
            sr.Write("Tempo: " + tempo + "\n");
            sr.Write("Key: " + key + "\n");
            sr.Write("Octave: " + octave + "\n");

            int numTracks = tracks.Count;
            sr.Write("Number of measures: " + measures.Count + "\n");
            sr.Write("Number of tracks: " + tracks.Count + "\n");
            sr.Write("HEADER - END\n");
            tabsStringImpl--;

            // Print measure information
            if (deep > 1)
            {
                tabsStringImpl++;
                sr.Write("MEASURES - BEGIN\n");
                for (int i = 0; i < measures.Count; i++)
                {
                    sr.Write("Measure #" + i + "\n");
                    sr.Write(measures.ElementAt(i) + "\n");
                }
                sr.Write("MEASURES - END\n");
                tabsStringImpl--;
            }

            // Print track information
            if (deep > 1)
            {
                tabsStringImpl++;
                sr.Write("TRACKS - BEGIN\n");

                for (int i = 0; i < numTracks; i++)
                {
                    sr.Write("Track #" + (i + 1) + "\n");
                    sr.Write(tracks.ElementAt(i).ToString() + "\n");
                }
                sr.Write("TRACKS - END\n");
                tabsStringImpl--;
            }

            // Print measure/track pairs information
            if (deep > 1)
            {
                tabsStringImpl++;
                sr.Write("MEASURE-TRACK PAIRS - BEING\n");
                for (int i = 0; i < measuresTracksPairs.Count; i++)
                {
                    sr.Write("Measure #" + ((i / numTracks) + 1) + " Track #" + (i % numTracks) + "\n");
                    sr.Write(measuresTracksPairs.ElementAt(i) + "\n");
                }
                sr.Write("MEASURE-TRACK PAIRS - END\n");
                tabsStringImpl--;
            }

            // Print MIDI channel information
            if (deep > 1)
            {
                tabsStringImpl++;
                sr.Write("MIDI CHANNELS - BEGIN\n");
                for (int i = 0; i < 4; i++)
                {
                    sr.Write("Port " + i + "\n");
                    for (int j = 0; j < 16; j++)
                    {
                        sr.Write("Channel: " + j + "\n");
                        sr.Write(channels[i * 16 + j] + "\n");
                    }
                }
                sr.Write("MIDI CHANNELS - END\n");
                tabsStringImpl--;
            }

            tabsStringImpl++;
            sr.Write("CHORD DIAGRAMS - BEGIN\n");
            sr.Write("Not yet implemented\n");
            sr.Write("CHORD DIAGRAMS - END\n");
            tabsStringImpl--;

            sr.Write("GP - END\n");

            return "";
        }

        #region Getters and setters

        // Returns the album name
        public string getAlbum()
        {
            return album;
        }

        // Returns the author of this piece
        public string getAuthorPiece()
        {
            return authorPiece;
        }

        // Returns the author of the song
        public string getAuthorSong()
        {
            return authorSong;
        }

        // Returns the channels used in this piece
        public GPMIDIChannel getChannels(int i)
        {
            return channels[i];
        }

        // Returns the copyright date
        public string getCopyright()
        {
            return copyright;
        }

        // Return the instructions associated with this piece
        public string getInstruction()
        {
            return instructions;
        }

        // Returns the interpret value of this piece
        public string getInterpret()
        {
            return interpret;
        }

        // Return the lyrics of this song
        public GPTrackLyrics getLyrics()
        {
            return lyrics;
        }

        // Return the notes of this song
        public string getNote()
        {
            return note;
        }

        // Return the subtitle of this piece
        public string getSubtitle()
        {
            return subtitle;
        }

        // Return the tempo of this piece
        public int getTempo()
        {
            return tempo;
        }

        // Return the title of the piece
        public string getTitle()
        {
            return title;
        }

        // Return the tracks of this piece
        public List<GPTrack> getTracks()
        {
            return tracks;
        }

        // Return the version that this piece was written for
        public string getVersion()
        {
            return version;
        }

        // Return the measures of this piece
        public List<GPMeasure> getMeasures()
        {
            return measures;
        }

        // Return the info about measure track pairs
        public List<GPMeasureTrackPair> getMeasuresTracksPairs()
        {
            return measuresTracksPairs;
        }

        // Return the number of tracks in this piece
        public int getNumTracks()
        {
            int res = 0;

            if (tracks != null)
                res = tracks.Count;

            return res;
        }

        // Return the number of mreasures in this piece
        public int getNumMeasures()
        {
            int res = 0;

            if (measures != null)
                res = measures.Count;

            return res;
        }

        // Return the pair of measures and tracks
        public GPMeasureTrackPair getTrackMeasurePair(int track, int measure)
        {
            return (GPMeasureTrackPair)measuresTracksPairs.ElementAt((measure - 1) * tracks.Count + track - 1);
        }

        // Set the album name
        public void setAlbum(string album)
        {
            this.album = album;
        }

        // Set the author of this piece
        public void setAuthorPiece(string authorPiece)
        {
            this.authorPiece = authorPiece;
        }

        // Set the author of the song
        public void setAuthorSong(string authorSong)
        {
            this.authorSong = authorSong;
        }

        // Set the channels of this piece
        public void setChannels(int i, GPMIDIChannel channels)
        {
            this.channels[i] = channels;
        }

        // Set the copright date
        public void setCopyright(string copyright)
        {
            this.copyright = copyright;
        }

        // Set the instructions of this piece
        public void setInstructions(string instruction)
        {
            this.instructions = instruction;
        }

        // Set the interpert for this piece
        public void setInterpret(string interpret)
        {
            this.interpret = interpret;
        }

        // Set the key of this piece
        public void setKey(GPKey key)
        {
            this.key = key;
        }

        // Set the lyrics of this song
        public void setLyrics(GPTrackLyrics lyrics)
        {
            this.lyrics = lyrics;
        }

        // Set the notes of this piece
        public void setNote(string note)
        {
            this.note = note;
        }

        // Set the octave for this piece
        public void setOctave(int octave)
        {
            this.octave = octave;
        }

        // Set the subtitles for this piece
        public void setSubtitle(string subtitle)
        {
            this.subtitle = subtitle;
        }

        // Set the tempo of this song'
        public void setTempo(int tempo)
        {
            this.tempo = tempo;
        }

        // Set the title of this song
        public void setTitle(string title)
        {
            this.title = title;
        }

        // Set the version of this song
        public void setVersion(string version)
        {
            this.version = version;
        }

        // Return the chords diagram for this song
        public List<GPChordDiagram> getChordsDiagrams()
        {
            return chordsDiagrams;
        }

        // Return the key this song is in
        public GPKey getKey()
        {
            return key;
        }

        // Returns a measure/track pair
        public GPMeasureTrackPair getMeasureTrackPair(int measure, int track)
        {
            return measuresTracksPairs.ElementAt((measure - 1) * tracks.Count + track - 1);
        }

        // Return the octave
        public int getOctave()
        {
            return octave;
        }
        #endregion
    }
}
