﻿using System.Collections.Generic;
using GuitarTrainer.Interfaces;
using GuitarTrainer.InterfaceImpl;

namespace GuitarTrainer.Sound
{
    /// <summary>
    /// An EventStream is the ordered list of PerformanceEvents on a specific virtual
    /// track, representing the behavior of a single string.
    /// </summary>
    class EventStream
    {
        #region Class variables
        /// <summary> The track to which this stream of events belongs </summary>
        private readonly ISongTrack track;

        /// <summary> The virtual track within 'track' that produced this stream of events </summary>
        private readonly int virtualTrackIndex;

        /// <summary>
        /// The first event in this event stream. It will typically not be null since tracks
        /// should being with an initialization event.
        /// </summary>
        private PerformanceEvent first;

        /// <summary>
        /// The last event in this event stream
        /// </summary>
        private PerformanceEvent last;

        /// <summary>
        /// The last event parsed
        /// </summary>
        private int lastEventEndOffset;

        /// <summary>
        /// The location of the last event
        /// </summary>
        private int lastEventLocation;

        /// <summary>
        /// The last note on the event parsed
        /// </summary>
        ISongEvent lastNoteOnEvent;
        #endregion

        public EventStream(ISongTrack songTrack, int v)
        {
            track = songTrack;
            virtualTrackIndex = v;
            first = null;
            last = null;
        }

        /// <summary>
        /// Add a list of SongEvents as PerformanceEvents to this event stream
        /// </summary>
        /// <param name="events">The list of events to add</param>
        /// <param name="location">The offset to apply to these events, in resolution units</param>
        public void addEvents(List<ISongEvent> events, int location)
        {
            foreach (ISongEvent se in events)
            {
                int baseTime = se.Time;
                ISongMessage message = se.Message;

                if (message is SongNoteOnMessage)
                {
                    closeOpenNote();
                    PerformanceEvent perfEvent = new PerformanceNoteEvent(this, baseTime + location, se);
                    connect(perfEvent);

                    lastNoteOnEvent = se;
                    SongNoteOnMessage noteOn = (SongNoteOnMessage)message;
                    lastEventEndOffset = baseTime + noteOn.Duration;
                    lastEventLocation = location;
                }
                else if (message is SongTieMessage)
                {
                    PerformanceEvent perfEvent = new PerformanceNoteEvent(this, baseTime + location, se);
                    connect(perfEvent);

                    SongTieMessage tie = (SongTieMessage)message;
                    lastEventEndOffset = baseTime + tie.Duration;
                    lastEventLocation = location;
                }
            }
        }

        private void closeOpenNote()
        {
            if (lastNoteOnEvent == null) return;

            SongNoteOnMessage noteOn = (SongNoteOnMessage)lastNoteOnEvent.Message;
            SongNoteOffMessage noteOff = new SongNoteOffMessage(noteOn);
            ISongEvent se = new SongEventImpl(lastEventEndOffset, noteOff)
                                {VirtualTrack = lastNoteOnEvent.VirtualTrack};
            PerformanceEvent perfEvent2 = new PerformanceNoteEvent(this, lastEventEndOffset + lastEventLocation, se);
            connect(perfEvent2);

            lastNoteOnEvent = null;
        }

        private void connect(PerformanceEvent perfEvent)
        {
            if (last != null)
                last.Next = perfEvent;
            
            last = perfEvent;
            
            if (first == null)
                first = perfEvent;
        }

        public void close()
        {
            closeOpenNote();
        }
    }
}
