﻿using GuitarTrainer.Interfaces;

namespace GuitarTrainer.Sound
{
    /// <summary>
    /// A IPerformance is an interface to a prepared version of a ISong. Implementations of
    /// this interface must be prepared to deliver PerformanceEvent objects to listeners
    /// when the music is playing. Implementations will also usually have to generate specific
    /// note data (such as MIDI messages or wav files).
    /// </summary>
    interface IPerformance
    {
        /// <summary>
        /// Set the frequency of the timer events in clicks per beat
        /// </summary>
        int TimerFrequency
        {
            set;
        }

        /// <summary>
        /// Generate initialization events (like controller initialization and patch
        /// selection) for the given ISongTrack. The implementor is responsible
        /// for creating the actual performed data as well as PerformanceTastInitEvent
        /// on virtual track 0 of the given ISongTrack.
        /// </summary>
        /// <param name="track">The ISongTrack to add</param>
        void initializeTrack(ISongTrack track);

        /// <summary>
        /// Generate timer events for the given measure, which will be at the specified location.
        /// It is up to the implementor to ensure that at least one event (corresponding to the
        /// measure start) is generated by this call.
        /// </summary>
        /// <param name="measure">The measure for which to add information</param>
        /// <param name="location">The time offset, me asured in timing pulses</param>
        void addTimerEvents(ISongMeasure measure, int location);

        /// <summary>
        /// Adds the given list of musical events to the performance. The particular performance
        /// implementation is responsible for adding these events as-is to the event stream
        /// for the particular index of track and virtual track. In addition, the performance 
        /// implementation must generate any necessary audio that will be played by the given
        /// player (like MIDI events or wav files).
        /// </summary>
        /// <param name="events">An event stream as PerformanceEvents to add</param>
        void addMusicalEvents(EventStream events);

        Tempo Tempo
        {
            get;
        }

        void setTimeSignature(int location, TimeSignature currentTimeSignature);
    }
}
