#region File Description
//-----------------------------------------------------------------------------
// DropObject.cs
//
// Copyright (C) Erik Ottosen. All rights reserved.
//-----------------------------------------------------------------------------
#endregion


using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Net;
using Microsoft.Xna.Framework.Storage;

namespace QuadBoom
{
    class DropObject
    {
        /// <summary>
        /// Each player's play field is filled with as many as 72 Drops, 
        /// which can be of one of five 'elements', or can be a trash icon.
        /// </summary>
        #region Fields
        public Texture2D sprite;
        public Vector2 currentPosition;
        public Vector2 targetPosition; // Only used for calculating Falling.
        public Rectangle locRectangle;
        public Vector2 center;
        public float rotation;
        public int arrayLocX;
        public int arrayLocY;
        public int motionFrameCounter; // this variable is also used after a piece has fallen to have the piece wobble a bit for a few frames.
        public int elimState; // used externally for the elimination logic.
        public DropElement element;
        public MotionState motionState;
        public Boolean turning; // Only used for active1 and active2. 
        // Turning does NOT equate to moving; it represents if motion is due to input.

        public enum DropElement
        {
            Air,
            Fire,
            Lightning,
            Water,
            Wood,
            Trash,
            Empty,
        }

        public enum MotionState
        {
            Still, // Not currently slated to move
            Falling, // Needs to fall to a new location.
            LandingSpin, // Piece has landed in the last few frames, and is rotationally 'wiggling'.
            Active1, // Is the 'top' piece of a moving pair.
            Active2, // Is the 'bottom' piece of a moving pair.
            NotOnField, // A 'safety' state which tells the piece that it isn't in play.
        }
        #endregion

        #region Field manipulations
        // Certain fields are not allowed to be manipulate.
        public Texture2D Sprite // Shouldn't be changed in normal play except for possible 'boom' sprites.
        {
            get { return sprite; }
            set { sprite = value;}
        }
        public Vector2 CurrentPosition
        {
            get { return currentPosition; }
            set { currentPosition = value; }
        }
        public Vector2 TargetPosition // Auto-recalculates the center.
        {
            get { return targetPosition; }
            set { 
                    targetPosition = value; 
                    center = new Vector2(sprite.Width / 2, sprite.Height / 2);
                }
        }
        public Vector2 Center // Shouldn't use the 'set', but the 'get' can matter for drawing...
        {
            get { return center; }
            set { center = value; }
        }
        public float Rotation
        {
            get { return rotation; }
            set { rotation = value; }
        }

        public int ArrayLocX
        {
            get { return arrayLocX; }
            set { arrayLocX = value; }
        }
        public int ElimState
        {
            get { return elimState; }
            set { elimState = value; }
        }
        public int ArrayLocY
        {
            get { return arrayLocY; }
            set { arrayLocY = value; }
        }
        public int MotionFrameCounter
        {
            get { return motionFrameCounter; }
            set { motionFrameCounter = value; }
        }
        public DropElement Element // Shouldn't be changed in normal play, but specials might want to change this.
        {
            get { return element; }
            set { element = value; }
        }
        public MotionState CurMotionState
        {
            get { return motionState; }
            set { motionState = value; }
        }
        public bool Turning
        {
            get { return turning; }
            set { turning = value; }
        }
        #endregion

        #region Constructors

        public DropObject(Texture2D inTexture, DropElement inElement, int inX, int inY, int targDimension)
        {
            rotation = 0.0f;
            currentPosition = Vector2.Zero; // Will update to actually set the proper drawing location soon...
            targetPosition = Vector2.Zero; // Will update to set the proper drawing location soon...
            locRectangle = new Rectangle((int)currentPosition.X, (int)currentPosition.Y, targDimension, targDimension);
            sprite = inTexture;
            center = new Vector2(sprite.Width / 2, sprite.Height / 2); // Used for calculating rotations.
            arrayLocX = inX;
            arrayLocY = inY;
            motionFrameCounter = 0;
            element = inElement;
            motionState = MotionState.Still;
            turning = false;
            elimState = 0;
        }


        #endregion

        #region Methods (including "Update" and draw handling)

        /// <summary>
        /// Recalculates the drawing position of this tile based on its Array locations.
        /// Should only need to be done when the Array locations change due to player input or the natural drop.
        /// </summary>
        public void RecalculatePositions(Rectangle playArea, int playAreaX, int playAreaY)
        {
            int tileWidth = playArea.Width / playAreaX;
            currentPosition.X = playArea.X + (tileWidth * arrayLocX-2);
            targetPosition.X = currentPosition.X;
            locRectangle.X = (int)currentPosition.X;
            currentPosition.Y = playArea.Y + ( tileWidth * (playAreaY-arrayLocY-3) );
            targetPosition.Y = currentPosition.Y;
            locRectangle.Y = (int)currentPosition.Y;
        }

        /// <summary>
        /// Calculates the target position of a tile based on it's Array locations, without touching the current.
        /// </summary>
        public void CalculateTargetPosition(Rectangle playArea, int playAreaX, int playAreaY)
        {
            targetPosition.X = playArea.X + ( (playArea.Width / playAreaX) * arrayLocX-2 );
            targetPosition.Y = playArea.Y + ((playArea.Width/playAreaX) * (playAreaY - arrayLocY - 3));
        }

        /// <summary>
        /// Instructs the piece to shift to the left by one space, and possibly also up or down by one.
        /// Has a variant with no ShiftFactor, that doesn't shift up or down.
        /// </summary>
        public void ShiftPieceLeft(Rectangle playAreaRect, int playAreaX, int playAreaY, int shiftFactor)
        {
            ArrayLocX--;
            if (shiftFactor >= 0) ArrayLocY++;
            else if (shiftFactor <= 0) ArrayLocY--;
            RecalculatePositions(playAreaRect, playAreaX, playAreaY);
        }
        public void ShiftPieceLeft(Rectangle playAreaRect, int playAreaX, int playAreaY)
        {
            ArrayLocX--;
            RecalculatePositions(playAreaRect, playAreaX, playAreaY);
        }

        /// <summary>
        /// Instructs the piece to shift to the left by one space, and possibly also up or down by one.
        /// Has a variant with no ShiftFactor, that doesn't shift up or down.
        /// </summary>
        public void ShiftPieceRight(Rectangle playAreaRect, int playAreaX, int playAreaY, int shiftFactor)
        {
            ArrayLocX++;
            if (shiftFactor >= 0) ArrayLocY++;
            else if (shiftFactor <= 0) ArrayLocY--;
            RecalculatePositions(playAreaRect, playAreaX, playAreaY);
        }
        public void ShiftPieceRight(Rectangle playAreaRect, int playAreaX, int playAreaY)
        {
            ArrayLocX++;
            RecalculatePositions(playAreaRect, playAreaX, playAreaY);
        }


        /// <summary>
        /// Instructs the piece to stop moving.
        /// </summary>
        public void StopPiece(Rectangle playAreaRect, int playAreaX, int playAreaY)
        {
            motionState = DropObject.MotionState.Still;
            RecalculatePositions(playAreaRect, playAreaX, playAreaY);
            motionFrameCounter = 0;
        }

        /// <summary>
        /// Instructs the piece to move down one tile, and reset its motionFrameCounter.
        /// </summary>
        public void StepPiece(Rectangle playAreaRect, int playAreaX, int playAreaY, int counterResetValue)
        {
            arrayLocY -= 1;
            RecalculatePositions(playAreaRect, playAreaX, playAreaY);
            motionFrameCounter = counterResetValue;
        }

        /// <summary>
        /// Instructs the piece to start falling.
        /// </summary>
        public void DropPiece(Rectangle playAreaRect, int playAreaX, int playAreaY, int dropDistance)
        {
            motionState = DropObject.MotionState.Still;
            arrayLocY -= dropDistance;
            //CalculateTargetPosition(playAreaRect, playAreaX, playAreaY);
            RecalculatePositions(playAreaRect, playAreaX, playAreaY);
            motionFrameCounter = 0;
        }

        public void subUpdate() // Updates this tile's state. Called by GamePlayScreen's Update method.
        {
            if (motionState == MotionState.Falling) // Falls are ALWAYS straight down.
            {
                float distToFall = currentPosition.Y - targetPosition.Y;
                if (distToFall < motionFrameCounter) // Piece will land at its target this frame.
                {
                    currentPosition.Y = targetPosition.Y;
                    locRectangle.Y = (int)targetPosition.Y;
                    if (motionFrameCounter % 2 != 0) motionFrameCounter++;
                    motionState = MotionState.LandingSpin;
                }
                else
                {
                    currentPosition.Y = currentPosition.Y - motionFrameCounter; // This makes the piece fall faster each frame until it lands.
                    locRectangle.Y = (int)currentPosition.Y;
                    motionFrameCounter++;
                }
            }
            else if (motionState == MotionState.LandingSpin)
            {
                if (motionFrameCounter % 4 == 0) rotation = 1.0f; 
                else if (motionFrameCounter % 4 == 2) rotation = -1.0f; 
                else { rotation = 0.0f; }
                motionFrameCounter--;
                if (motionFrameCounter == 0) { motionState = MotionState.Still; }
            }

            else if (motionState == MotionState.NotOnField)
            {
                // Here largely as a safety.
                Console.Write("NotOnField called on a subUpdate!");
            }
        }

        public void Draw(SpriteBatch inBatch)
        {
            // Pretty quick and easy, because this item should only draw in response to being told to by the game...
            // ... So it can draw inside GameScreen.Draw's SpriteBatch. Thus the lack of a Begin or End call.

            inBatch.Draw(sprite, // The texture
                locRectangle, // The location vector
                null, // The Source Rectangle - lets you draw just a part of an image, and have all sprites in one file.
                // Null draws the entire image, plain and simple.
                Color.White, // The sprites' color tint - White is unmodified.
                rotation, // How much to rotate it by, in Radians.
                center, // Origin of Rotation - normally you'd want it at the sprite's center, naturally.
                // ( 0.5f + rotation/8), // How much to scale it by. Designed to warp a bit here to make the sprite 'wobble'.
                SpriteEffects.None, // Lets you flip the sprite Horizontally, Vertically, or neither. (Not both? :<)
                0.1f); // Which layer the sprite goes on. Lower depth = foreground, higher = background. (0 to 1 range.)
                // Set to 0.1 here to allow the AreaFrame to override. "f" specifies float.

        }

        #endregion
    }
}