#region File Description
//-----------------------------------------------------------------------------
// GameplayScreen.cs
//
// Microsoft XNA Community Game Platform
// Copyright (C) Microsoft Corporation. All rights reserved.
// Modified by Erik Ottosen for QuadBoom, Copyright (C) Erik Ottosen 2010.
//-----------------------------------------------------------------------------
#endregion

#region Using Statements
using System;
using System.Threading;
using System.Diagnostics;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
#endregion

namespace QuadBoom
{
    /// <summary>
    /// This screen is where the actual gameplay goes.
    /// This is where the majority of my work is, naturally.
    /// In the demonstration, there was an unrelated sample.
    /// </summary>
    class GameplayScreen : GameScreen
    {
        #region Non-Texture Fields

        //GraphicsDeviceManager graphics;
        private ContentManager content;
        private SpriteFont gameFont;
        private SpriteBatch spriteBatch;
        /*
        // Pieces from the sample code.
        Vector2 playerPosition = new Vector2(100, 100);
        Vector2 enemyPosition = new Vector2(100, 100);
        */
        // Graphical bits.
        private static int tileSizeX = 48, tileSizeY = 48, tileSpaceX = 50, tileSpaceY = 50;
        private Vector2 tileSize = new Vector2(tileSizeX, tileSizeY);
        private Vector2 tileSpace = new Vector2(tileSpaceX, tileSpaceY); // Will be De-Magic-Number'd later.
        private Vector2 playAreaSize;

        // Gameplay fields. (Many double as having a graphical purpose.)
        private int playAreaX = 6; // Number of spaces, each holding one tile, in the play area's width.
        private int playAreaY = 12; // How many spaces high the play area is.
        private int framesPerMotion = 15; // How many frames should pass before an active piece drops by one tile.
        private DropObject[,] playField; // The actual contents of the field are updated in this Array for checking for clearing.
        private DropObject active1;
        private DropObject active2;

        KeyboardState previousKeyboardState;
        GamePadState previousGamepadState;

        Random random;

        public enum DropElement
        {
            Air,
            Fire,
            Lightning,
            Water,
            Wood,
            Trash,
            Empty,
        }

        #endregion

        #region Textures

        private Texture2D playArea;
        private Texture2D playAreaFrame;
        private Texture2D[] elementSprites;
        private Rectangle playAreaRect; // Helper for the Play Area. Needed for resolution handling.
        private Rectangle playAreaFrameRect;
        // We already have a SpriteBatch from the ScreenManager.

        #endregion

        #region Initialization


        /// <summary>
        /// Constructor.
        /// </summary>
        public GameplayScreen()
        {
            TransitionOnTime = TimeSpan.FromSeconds(1.5);
            TransitionOffTime = TimeSpan.FromSeconds(0.5);
            playField = new DropObject[playAreaX , playAreaY];
            // The playfield area is meant to be 600*300 ideally.
            elementSprites = new Texture2D[7]; // Number of possible results for Element = 7 - 0-4 for elements, 5 for trash, 6 for empty.
        }


        /// <summary>
        /// Load graphics content for the game.
        /// </summary>
        public override void LoadContent()
        {
            if (content == null)
                content = new ContentManager(ScreenManager.Game.Services, "Content");

            gameFont = content.Load<SpriteFont>("gamefont");
            playArea = content.Load<Texture2D>("ScreenItems/PlayArea");
            playAreaFrame = content.Load<Texture2D>("ScreenItems/PlayAreaFrame");
            elementSprites[0] = content.Load<Texture2D>("Drops/Droplet-Air");
            elementSprites[1] = content.Load<Texture2D>("Drops/Droplet-Fire");
            elementSprites[2] = content.Load<Texture2D>("Drops/Droplet-Lightning");
            elementSprites[3] = content.Load<Texture2D>("Drops/Droplet-Water");
            elementSprites[4] = content.Load<Texture2D>("Drops/Droplet-Wood");
            elementSprites[5] = content.Load<Texture2D>("Drops/Droplet-Trash");

            /* Don't need to initialize to nulls...
            foreach (DropObject i in playField)
            {
                i = null; // Field starts with nothing in it.
            }
             */ 

            int playAreaHeight = determineScaleY(); 
            playAreaSize = new Vector2( playAreaHeight/2.4f, playAreaHeight/1.2f );
            tileSpace = new Vector2(playAreaSize.X/playAreaX, playAreaSize.Y/playAreaY);
            tileSize = new Vector2(tileSpace.X-1, tileSpace.Y-1);
            //playAreaRect = new Rectangle(playAreaHeight/10, playAreaHeight/10, (int)playAreaSize.X, (int)playAreaSize.Y);
            playAreaRect = new Rectangle((int)ScreenManager.GraphicsDevice.Viewport.TitleSafeArea.Left+(int)tileSpace.X*2, (int)ScreenManager.GraphicsDevice.Viewport.TitleSafeArea.Top+(int)tileSpace.Y*4, (int)playAreaSize.X, (int)playAreaSize.Y);
            playAreaFrameRect = new Rectangle(playAreaRect.Left - playAreaRect.Width / 6, playAreaRect.Top - playAreaRect.Height / 12, playAreaRect.Width*4/3, playAreaRect.Height*7/6);
            // No sprite for 'empty' right now...
            // This is used to stall the load screen for the moment... This'll matter more once more content is added.
            Thread.Sleep(80);

            // Initialize the randomizer. Currently doesn't provide a seed source, defaulting to the system timer.
            random = new Random();

            // Spawn first pieces. In later builds, may want a Tetris Guideline-style randomizer.
            active1 = generateNewPiece(playAreaX / 2 + 0, playAreaY + 1);
            active2 = generateNewPiece(playAreaX / 2 + 1, playAreaY + 1);
            // Note that this will cause both Actives to immediately drop into the play Area on the first frame.

            // once the load has finished, we use ResetElapsedTime to tell the game's
            // timing mechanism that we have just finished a very long frame, and that
            // it should not try to catch up.
            ScreenManager.Game.ResetElapsedTime();
        }


        /// <summary>
        /// Unload graphics content used by the game.
        /// </summary>
        public override void UnloadContent()
        {
            content.Unload();
        }


        #endregion

        #region Methods
        public int determineScaleY()
        {
            // Figures out what the Y area we'll be working with is - the Y scale ALONE defines the play area's size.
            int safeArea = ScreenManager.GraphicsDevice.Viewport.TitleSafeArea.Bottom - ScreenManager.GraphicsDevice.Viewport.TitleSafeArea.Top;
            int idealSize = playArea.Height;
            if (safeArea < idealSize ) { return safeArea; }
            else return idealSize;
        }

        // TODO: Once I've drawn up the center area and have two-player support in, 
        // a determineScaleX method will be wanted for calculating placements.

        public DropObject getComparator(DropObject inObj)
        {
            // Comparator is the tile below the current piece, to determine if that piece can drop or not.
            DropObject comparator;
            if (inObj.ArrayLocY > playAreaY - 1) comparator = playField[inObj.ArrayLocX, playAreaY - 1];
            else if (inObj.ArrayLocY == 0) comparator = playField[inObj.ArrayLocX, 0]; // Won't be used, but put in as a safety.
            else comparator = playField[inObj.ArrayLocX, inObj.ArrayLocY - 1];

            return comparator;
        }

        public int dropDistance(DropObject inObj)
        {   // Determines how far a piece needs to fall before it stops.
            // Basically, keep checking tiles until one is not empty, or the floor, is found.
            int fallDist = 0;
            while ( isSpaceBelow(inObj.arrayLocX, inObj.ArrayLocY-fallDist) ) // In laymen's terms, is this space open?
            {
                fallDist++;
            }
            return fallDist;
        }

        /// <summary>
        /// Determines if there is an empty space below input [x,y], returning TRUE if so.
        /// </summary>
        /// 
        public bool isSpaceBelow(int x, int y)
        {
            if (y > 0 && playField[x, y-1] == null) return true;
            else return false;
        }


        /// <summary>
        /// Tests the board for pieces to remove from play (due to being in a group of four or more of the same color).
        /// </summary>
        public int eliminationLogic()
        {
            int[] elimStateFreq = new int[(playAreaX * playAreaY) + 1]; // Convieniently initializes to all zeroes in the C# spec!
            int quadboomsCount = 0;
            int quadboomsPiecesDropped = 0;
            int targElimState;

            // Spider through every piece on the board, determining the size of each contiguous group of one element.
            //Trace.WriteLine("Beginning Elimination Logic - scanning for spidering");
            for (int i = 0; i < playAreaX; i++)
            {
                for (int j = 0; j < playAreaY; j++)
                {
                    if (playField[i, j] != null && playField[i, j].ElimState == 0)
                    {
                        playField[i, j].ElimState = i * playAreaY + j + 1; // this nicely indexes each number.
                        targElimState = playField[i, j].ElimState;
                        elimStateFreq[targElimState]++;
                        //Trace.WriteLine("Spidering on position " + i + "," + j + ".");
                        elimStateFreq[targElimState] += eliminationSpider(i, j, targElimState);
                        //Trace.WriteLine("Group " + targElimState + " has size " + elimStateFreq[targElimState] + ".");
                    }
                }
            }

            // Find each group of size four, and eliminate all pieces from it.
            //Trace.WriteLine("Beginning Elimination Logic - Group Checking");
            for (int k = 1; k < elimStateFreq.Length - 1; k++)
            {
                //if (elimStateFreq[k] > 0) { Trace.WriteLine("Group " + k + " tests at " + elimStateFreq[k] + "."); }
                if (elimStateFreq[k] >= 4)
                {
                    quadboomsCount++;
                    quadboomsPiecesDropped += destroyPiecesOfState(k);
                }
            }

            // Reset ALL elimStates to 0. No, a foreach loop won't work here - as sometimes there will be zero DropObjects in the field.
            for (int i = 0; i < playAreaX; i++)
            {
                for (int j = 0; j < playAreaY; j++)
                {
                    if (playField[i, j] != null) { playField[i, j].ElimState = 0; }
                }
            }

            if (quadboomsCount != 0)
            {
               quadboomsPiecesDropped += eliminationLogic();
                // YES we have to do it all again if any pieces went boom - because more might go boom in response!
            }

            return quadboomsPiecesDropped;
        }

        /// <summary>
        /// Recursively checks every piece around the target piece; if that piece is of the same color as the target,
        /// its ElimState is set to equal the target's, and it is in turn spidered.
        /// Returns the number of pieces whose ElimState was modified by this.
        /// </summary>
        /// 
        public int eliminationSpider(int x, int y, int targElimState)
        {
            int totalChanged = 0;
            if (x >= 1)
            {
                if (playField[x-1,y] != null && playField[x, y].Element == playField[x - 1, y].Element &&
                    playField[x, y].ElimState != playField[x - 1, y].ElimState )
                {
                    //Trace.WriteLine("Testing to left of" + x + "," + y + ".");
                    playField[x - 1, y].ElimState = targElimState;
                    totalChanged++;
                        totalChanged += eliminationSpider(x - 1, y, targElimState);
                }
            }
            if (x < playAreaX-1)
            {
                if (playField[x+1, y] != null && playField[x, y].Element == playField[x + 1, y].Element &&
                    playField[x, y].ElimState != playField[x + 1, y].ElimState )
                {
                    //Trace.WriteLine("Testing to right of" + x + "," + y + ".");
                    playField[x + 1, y].ElimState = targElimState;
                    totalChanged++;
                    totalChanged += eliminationSpider(x + 1, y, targElimState);
                }
            }
            if (y >= 1)
            {

                if (playField[x, y-1] != null && playField[x, y].Element == playField[x, y - 1].Element
                    && playField[x, y].ElimState != playField[x, y - 1].ElimState)
                {
                    //Trace.WriteLine("Testing above" + x + "," + y + ".");
                    playField[x, y - 1].ElimState = targElimState;
                    totalChanged++;

                    totalChanged += eliminationSpider(x, y - 1, targElimState);
                }
            }
            if (y < playAreaY-1)
            {
                if (playField[x, y+1] != null && playField[x, y].Element == playField[x, y + 1].Element
                    && playField[x, y].ElimState != playField[x, y + 1].ElimState)
                {
                    //Trace.WriteLine("Testing below" + x + "," + y + ".");
                    playField[x, y + 1].ElimState = targElimState;
                    totalChanged++;
                    totalChanged += eliminationSpider(x, y + 1, targElimState);
                }
            }
            return totalChanged;
        }

        /// <summary>
        /// Executed in the middle of Elimination Logic if a group of size four or greater is found.
        /// Per the name, all pieces of that group (identified by its ElimState value) are destroyed.
        /// Could also be used for a lot of special moves, and to rapidly erase the entire board outside of Logic
        /// by running it with value 0 outside of the Logic.
        /// </summary>
        /// 
        public int destroyPiecesOfState(int destroyVal)
        {
            int piecesDestroyed = 0;
            bool[] dropField = new bool[playAreaX]; 
            foreach (DropObject i in playField)
            {
                if (i != null && i.ElimState == destroyVal)
                {
                    piecesDestroyed++;
                    playField[i.ArrayLocX, i.ArrayLocY] = null;
                    dropField[i.ArrayLocX] = true;
                }
            }

            for (int h = 0; h < playAreaX - 1; h++)
            {
                if (dropField[h])
                {
                    calcDrops(h);
                }
            }
            return piecesDestroyed;
        }

        /// <summary>
        /// Checks every space in the specified row, from the bottom.
        /// If that space is filled, run DropPiece on it by a value equal to the number of empty spaces found.
        /// </summary>
        public void calcDrops(int x)
        {
            int dropCounter = 0;
            for (int i = 0; i < playAreaY - 1; i++)
            {
                if (playField[x, i] == null)
                {
                    dropCounter++;
                }
                else if (dropCounter > 0)
                {
                    playField[x,i].DropPiece(playAreaRect, playAreaX, playAreaY, dropCounter);
                    playField[x, i - dropCounter] = playField[x, i];
                    playField[x, i] = null;
                }
            }
        }

        /// <summary>
        /// Processes general motion for both active pieces.
        /// Controller inputs can also cause motion (directly AND indirectly) in varying ways.
        /// Returns "true" if either piece is still in motion (either due to being under player control, or falling).
        /// </summary>
        public bool processMotion()
        {
            if (active1.CurMotionState == DropObject.MotionState.Falling || active2.CurMotionState == DropObject.MotionState.Falling)
            {
                active1.subUpdate();
                active2.subUpdate();
                return true;
            }

            if (active1.MotionFrameCounter <= 0) 
                // We always use 1's motion counter to determine if the pieces need to drop, 
                // as a convienience measure. (Note though that the MFC is ALSO used for effects for pieces that are not active!)
            {
                DropObject comparator1 = getComparator(active1);
                DropObject comparator2 = getComparator(active2);

                if (active1.ArrayLocX == active2.ArrayLocX) // Both active pieces share a row. We only need to check the bottom one.
                { // Might be able to refactor down in size further in later builds.
                    if (active1.ArrayLocY > active2.ArrayLocY)
                    { // active2 is the bottom piece. Test it.
                        if (active2.ArrayLocY == 0)
                        {
                            active1.StopPiece(playAreaRect, playAreaX, playAreaY);
                            playField[active1.ArrayLocX, active1.ArrayLocY] = active1;
                            active2.StopPiece(playAreaRect, playAreaX, playAreaY);
                            playField[active2.ArrayLocX, active2.ArrayLocY] = active2;
                            return false;
                        }
                        else if (comparator2 != null)
                        {
                            active1.StopPiece(playAreaRect, playAreaX, playAreaY);
                            playField[active1.ArrayLocX, active1.ArrayLocY] = active1;
                            active2.StopPiece(playAreaRect, playAreaX, playAreaY);
                            playField[active2.ArrayLocX, active2.ArrayLocY] = active2;
                            return false;
                        }
                        else
                        { // active2 has nothing below it. Drop both pieces by one tile.
                            active1.StepPiece(playAreaRect, playAreaX, playAreaY, framesPerMotion);
                            active2.StepPiece(playAreaRect, playAreaX, playAreaY, framesPerMotion);
                            return true;
                        }
                    }
                    else
                    { // active1 is the bottom piece. Test it.
                        if (active1.ArrayLocY == 0)
                        {
                            active1.StopPiece(playAreaRect, playAreaX, playAreaY);
                            playField[active1.ArrayLocX, active1.ArrayLocY] = active1;
                            active2.StopPiece(playAreaRect, playAreaX, playAreaY);
                            playField[active2.ArrayLocX, active2.ArrayLocY] = active2;
                            return false;
                        }
                        else if (comparator1 != null)
                        {
                            active1.StopPiece(playAreaRect, playAreaX, playAreaY);
                            playField[active1.ArrayLocX, active1.ArrayLocY] = active1;
                            active2.StopPiece(playAreaRect, playAreaX, playAreaY);
                            playField[active2.ArrayLocX, active2.ArrayLocY] = active2;
                            return false;
                        }
                        else
                        { // active1 has nothing below it. Drop both pieces by one tile.
                            active1.StepPiece(playAreaRect, playAreaX, playAreaY, framesPerMotion);
                            active2.StepPiece(playAreaRect, playAreaX, playAreaY, framesPerMotion);
                            return true;
                        }
                    }
                }

                else if (active1.ArrayLocY == 0)
                { // The two pieces are both on the bottom row. Stop both as-is.
                    active1.StopPiece(playAreaRect, playAreaX, playAreaY);
                    playField[active1.ArrayLocX, active1.ArrayLocY] = active1;
                    active2.StopPiece(playAreaRect, playAreaX, playAreaY);
                    playField[active2.ArrayLocX, active2.ArrayLocY] = active2;
                    return false;
                }

                else
                { // The two pieces are on different columns, but the same row. Both need to be tested, 
                    // then if one freezes while the other doesn't, that one needs to go into a Falling state.
                    bool isOneFrozen = false, isTwoFrozen = false;
                    if (comparator1 != null)
                    {
                        isOneFrozen = true;
                        active1.StopPiece(playAreaRect, playAreaX, playAreaY);
                        playField[active1.ArrayLocX, active1.ArrayLocY] = active1;
                    }
                    if (comparator2 != null)
                    {
                        isTwoFrozen = true;
                        active2.StopPiece(playAreaRect, playAreaX, playAreaY);
                        playField[active2.ArrayLocX, active2.ArrayLocY] = active2;
                    }
                    if (isOneFrozen && !isTwoFrozen)
                    { // Piece Two needs to drop. Also, return true.
                        active2.DropPiece(playAreaRect, playAreaX, playAreaY, dropDistance(active2));
                        playField[active2.ArrayLocX, active2.ArrayLocY] = active2;
                        return true;
                    }
                    else if (!isOneFrozen && isTwoFrozen)
                    { // Piece one needs to drop. Also, return true.
                        active1.DropPiece(playAreaRect, playAreaX, playAreaY, dropDistance(active1));
                        playField[active1.ArrayLocX, active1.ArrayLocY] = active1;
                        return true;
                    }
                    else if (!isOneFrozen && !isTwoFrozen)
                    { // Neither piece is frozen, both need to be stepped down one.
                        active1.StepPiece(playAreaRect, playAreaX, playAreaY, framesPerMotion);
                        active2.StepPiece(playAreaRect, playAreaX, playAreaY, framesPerMotion);
                        return true;
                    }
                    else
                    { // Both pieces are still now. Return false.
                        return false;
                    }
                 }
            }

            else
            { // The MotionFrameCounter isn't at 0 yet. Reduce it by one.
                active1.MotionFrameCounter--;
                if (active1.MotionFrameCounter < 0) active1.MotionFrameCounter = 0; // Thrown in as a safety.
                // If I want to get fancy in a later build, throw in 'smooth' position shifting by manipulating CurrentPosition. Ignoring for now.
                return true;
            }
        }


        public DropObject generateNewPiece(int inX, int inY)
        {
            int elementPick = random.Next(0,4); // Selects one of the 5 elements.
            int targDimension = playAreaRect.Width/playAreaX;
            DropObject outObj = new DropObject(elementSprites[5], DropObject.DropElement.Trash, inX, inY, targDimension); // Safety
            switch (elementPick)
            {
                case 0: // Air
                    outObj = new DropObject(elementSprites[0], DropObject.DropElement.Air, inX, inY, targDimension);
                    break;
                case 1: // Fire
                    outObj = new DropObject(elementSprites[1], DropObject.DropElement.Fire, inX, inY, targDimension);
                    break;
                case 2: // Lightning
                    outObj = new DropObject(elementSprites[2], DropObject.DropElement.Lightning, inX, inY, targDimension);
                    break;
                case 3: // Water
                    outObj = new DropObject(elementSprites[3], DropObject.DropElement.Water, inX, inY, targDimension);
                    break;
                case 4: // Wood
                    outObj = new DropObject(elementSprites[4], DropObject.DropElement.Wood, inX, inY, targDimension);
                    break;
            }
            outObj.MotionFrameCounter = framesPerMotion;
            return outObj;
        }

        public bool isButtonNewlyDown(GamePadState curGamePadState, GamePadState prevGamePadState, Buttons button)
        {
            if (curGamePadState.IsButtonUp(button)) { return false; }
            else {
                if (prevGamePadState.IsButtonDown(button)) { return false; }
                else { return true; }
            }

        }

        #endregion

        #region Update and Draw


        /// <summary>
        /// Updates the state of the game. This method checks the GameScreen.IsActive
        /// property, so the game will stop updating when the pause menu is active,
        /// or if you tab away to a different application.
        /// </summary>
        public override void Update(GameTime gameTime, bool otherScreenHasFocus,
                                                       bool coveredByOtherScreen)
        {
            base.Update(gameTime, otherScreenHasFocus, coveredByOtherScreen);

            if (IsActive)
            {

                bool anyMotion = false;

                foreach (DropObject i in playField) // This should convieniently skip nulls!
                {
                    if (i == null || i.CurMotionState == DropObject.MotionState.Still) continue;
                    else if (i.CurMotionState == DropObject.MotionState.NotOnField)
                    {
                        playField[i.ArrayLocX, i.ArrayLocY] = null;
                        // Need to figure out a way to remove i so that it is put up for Garbage Collection for certain.
                    }
                    else
                    {
                        anyMotion = true;
                        i.subUpdate();
                    }
                }

                // Active piece motion is processed here.
                anyMotion = (processMotion() || anyMotion);

                if (!anyMotion)
                {
                    // No pieces are moving.
                    // TODO: Insert code for quad scanning here.
                    eliminationLogic();

                    // Fake Game Over condition - wipes the board if the center pieces wouldn't fit in.
                    if (playField[playAreaX / 2, playAreaY - 1] != null || playField[(playAreaX / 2) + 1, playAreaY - 1] != null)
                    {
                        destroyPiecesOfState(0);
                    }
                    // Spawn new pieces. In later builds, may want a Tetris Guideline-style randomizer.
                    active1 = generateNewPiece(playAreaX/2+0,playAreaY+1);
                    active2 = generateNewPiece(playAreaX/2+1,playAreaY+1);
                    // Note that this will cause both Actives to immediately drop into the play Area on the next frame.

                }
                // This when any special moves or trash drops get processed.

            }
        }

        /// <summary>
        /// Lets the game respond to player input. Unlike the Update method,
        /// this will only be called when the gameplay screen is active.
        /// Mass Update necessary for 2nd player support.
        /// </summary>
        public override void HandleInput(InputState input)
        {
            if (input == null)
                throw new ArgumentNullException("input");

            // Look up inputs for the active player profile.
            int playerIndex = (int)ControllingPlayer.Value;

            KeyboardState keyboardState = input.CurrentKeyboardStates[playerIndex];
            if (previousKeyboardState == null) previousKeyboardState = keyboardState;
            GamePadState gamePadState = input.CurrentGamePadStates[playerIndex];
            if (previousGamepadState == null) previousGamepadState = gamePadState;

            // The game pauses either if the user presses the pause button, or if
            // they unplug the active gamepad. This requires us to keep track of
            // whether a gamepad was ever plugged in, because we don't want to pause
            // on PC if they are playing with a keyboard and have no gamepad at all!
            // (An alternate way to do this would be to fit the test into an #if !XBOX Block.)
            bool gamePadDisconnected = !gamePadState.IsConnected &&
                                       input.GamePadWasConnected[playerIndex];

            if (input.IsPauseGame(ControllingPlayer) || gamePadDisconnected)
            {
                ScreenManager.AddScreen(new PauseMenuScreen(), ControllingPlayer);
            }

            else if (active1 != null && active2 != null) // This was quickly coded in as a breakage prevention measure.
            {
                if (!active1.Turning && !active2.Turning)
                {
                    Vector2 thumbstick = gamePadState.ThumbSticks.Left;
                    //if (keyboardState.IsKeyDown(Keys.Left) | gamePadState.IsButtonDown(Buttons.DPadLeft) | thumbstick.X < 0)
                    if (isButtonNewlyDown(gamePadState, previousGamepadState, Buttons.DPadLeft))
                    {
                        // Push both active pieces left, unless they're on the edge of the field.
                        if (active1.ArrayLocX != 0 && active2.ArrayLocX != 0)
                        {
                            active1.ShiftPieceLeft(playAreaRect, playAreaX, playAreaY, 0);
                            active2.ShiftPieceLeft(playAreaRect, playAreaX, playAreaY, 0);
                        }
                    }
                    //else if (keyboardState.IsKeyDown(Keys.Right) | gamePadState.IsButtonDown(Buttons.DPadRight) | thumbstick.X > 0)
                    else if (isButtonNewlyDown(gamePadState, previousGamepadState, Buttons.DPadRight))
                    {  // Else if used to prevent motion conflict in the case of, say, the analog shift saying left but the Dpad right.
                        // Push both active pieces right.
                        if (active1.ArrayLocX < playAreaX-1 && active2.ArrayLocX < playAreaX-1)
                        {
                                active1.ShiftPieceRight(playAreaRect, playAreaX, playAreaY, 0);
                                active2.ShiftPieceRight(playAreaRect, playAreaX, playAreaY, 0);
                        }
                    }
     
                        //else if (keyboardState.IsKeyDown(Keys.Up) | gamePadState.IsButtonDown(Buttons.DPadUp) | thumbstick.Y > 0.5)
                    else if (isButtonNewlyDown(gamePadState, previousGamepadState, Buttons.DPadUp))
                    {
                        // TODO: Instantly drop pieces. May want to have this be optional in later builds.
                    }
                    //else if (keyboardState.IsKeyDown(Keys.Down) | gamePadState.IsButtonDown(Buttons.DPadDown) | thumbstick.Y > 0.5)
                    else if (isButtonNewlyDown(gamePadState, previousGamepadState, Buttons.DPadDown))
                    {
                        // Drop the pieces faster than usual by zeroing out the Motion Frame Counter for active1.
                        active1.MotionFrameCounter = 0;
                    }
                    //if (keyboardState.IsKeyDown(Keys.Z) | gamePadState.IsButtonDown(Buttons.A))
                    if (isButtonNewlyDown(gamePadState, previousGamepadState, Buttons.A))
                    {
                        // Rotate pieces COUNTERclockwise about active2.\
                        if (active1.ArrayLocX + 1 == active2.ArrayLocX)
                        {
                            active1.ShiftPieceRight(playAreaRect, playAreaX, playAreaY, -1);
                        }
                        else if (active1.ArrayLocX - 1 == active2.ArrayLocX)
                        {
                            active1.ShiftPieceLeft(playAreaRect, playAreaX, playAreaY, 1);
                        }
                        else if (active1.ArrayLocY + 1 == active2.ArrayLocY)
                        {
                            if (active1.ArrayLocX == playAreaX - 1)
                            { // "wall kick" shifts the pieces in if they're on the edge.
                                active1.ShiftPieceLeft(playAreaRect, playAreaX, playAreaY);
                                active2.ShiftPieceLeft(playAreaRect, playAreaX, playAreaY);
                            }
                            active1.ShiftPieceRight(playAreaRect, playAreaX, playAreaY, 1);
                        }
                        else if (active1.ArrayLocY - 1 == active2.ArrayLocY)
                        {
                            if (active1.ArrayLocX == 0)
                            { // "wall kick" shifts the pieces in if they're on the edge.
                                active1.ShiftPieceRight(playAreaRect, playAreaX, playAreaY);
                                active2.ShiftPieceRight(playAreaRect, playAreaX, playAreaY);
                            }
                            active1.ShiftPieceLeft(playAreaRect, playAreaX, playAreaY, -1);
                        }
                    }
                    //if (keyboardState.IsKeyDown(Keys.X) | gamePadState.IsButtonDown(Buttons.B))
                    else if (isButtonNewlyDown(gamePadState, previousGamepadState, Buttons.B))
                    {
                        // Rotate pieces clockwise about active2.
                        if (active1.ArrayLocX + 1 == active2.ArrayLocX)
                        {
                            active1.ShiftPieceRight(playAreaRect, playAreaX, playAreaY, 1);
                        }
                        else if (active1.ArrayLocX - 1 == active2.ArrayLocX)
                        {
                            active1.ShiftPieceLeft(playAreaRect, playAreaX, playAreaY, -1);
                        }
                        else if (active1.ArrayLocY + 1 == active2.ArrayLocY)
                        {
                            if (active1.ArrayLocX == 0)
                            { // "wall kick" shifts the pieces in if they're on the edge.
                                active1.ShiftPieceRight(playAreaRect, playAreaX, playAreaY);
                                active2.ShiftPieceRight(playAreaRect, playAreaX, playAreaY);
                            }
                            active1.ShiftPieceLeft(playAreaRect, playAreaX, playAreaY, 1);
                        }
                        else if (active1.ArrayLocY - 1 == active2.ArrayLocY)
                        {
                            if (active1.ArrayLocX == playAreaX-1)
                            { // "wall kick" shifts the pieces in if they're on the edge.
                                active1.ShiftPieceLeft(playAreaRect, playAreaX, playAreaY);
                                active2.ShiftPieceLeft(playAreaRect, playAreaX, playAreaY);
                            }
                            active1.ShiftPieceRight(playAreaRect, playAreaX, playAreaY, -1);
                        }
                    }
                }
            }

            previousGamepadState = gamePadState;
            previousKeyboardState = keyboardState;

        }


        /// <summary>
        /// Draws the gameplay screen.
        /// </summary>
        public override void Draw(GameTime gameTime)
        {
            SpriteBatch spriteBatch = ScreenManager.SpriteBatch;
            // The demo game uses a Blue BG as an arbitrary sample.
            //ScreenManager.GraphicsDevice.Clear(ClearOptions.Target, Color.Firebrick, 0, 0);

            spriteBatch.Begin(SpriteBlendMode.AlphaBlend, SpriteSortMode.BackToFront, SaveStateMode.None);
            Vector2 fieldCenter = new Vector2(playAreaRect.Width/2, playAreaRect.Height/2);
            // Draw the Play Area. For a full definition of what everything means, check DropObject.cs' draw method.
            spriteBatch.Draw(playArea, playAreaRect, null, Color.White, 0.0f, fieldCenter, SpriteEffects.None, 1);
            // TODO: Draw all icons.
            foreach (DropObject i in playField)
            {
                if (i != null) i.Draw(spriteBatch);
            }
            if (active1 != null) active1.Draw(spriteBatch);
            if (active2 != null) active2.Draw(spriteBatch);

            // Draw the Play Area FRAME last (so that it goes over the icons.)
            spriteBatch.Draw(playAreaFrame, playAreaFrameRect, null, Color.White, 0.0f, fieldCenter, SpriteEffects.None, 1);

            spriteBatch.End();
            // If the game is transitioning on or off, fade it out to black.
            if (TransitionPosition > 0)
                ScreenManager.FadeBackBufferToBlack(255 - TransitionAlpha);
        }


        #endregion
    }
}
