package objects.AI;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import utils.MoreMath;

/**
    The EvolutionGenePool class keeps track of a collection of
    Brains. When an EvolutionBot is created, it requests a
    Brain from the gene pool. The Brain is either one of the
    brains in the collection, or a mutation of one of the brains.
    Only the top brains are collected, ranked by the amount of
    damage a brain's bot caused. There for, only the best brains
    reproduce.
    
    Modified from code in "Developing Games in Java" by David Brakeen
*/

public class EvolutionGenePool 
{
    private static final int NUMBER_OF_TOP_BRAINS = 3;
    private static final int NUMBER_OF_TOTAL_BRAINS = 9;

    private class BrainStats extends Instinct implements Comparable 
    {
    	final int RATE_OF_LEARNING = 2;
    	final double CHANGE_IN_LEARNING = RATE_OF_LEARNING * MoreMath.gausian();
        long totalDamageCaused = 0;
        long totalTimeLived = 0;
        int generation;
        
        // mutation rate for each battle state
        public double mutationRun;
        public double mutationCharge;
        public double mutationRanged;
        public double mutationTrap;
        public double mutationEnchant;

        public BrainStats() {}

        public BrainStats(BrainStats brain) {
            super(brain);
            this.generation = brain.generation;
        }


        /**
         * The fitness function is the total damage caused plus a fraction of the total damage caused.
         * The fraction is the total damage caused multiplied by the percentage of the maximum amount 
         * of time to live the AI actually spent alive.  That way, if the AI lives the maximum amount
         * of time, but does no damage, it's fitness wont out-weight those that did more damage, but
         * lived a shorter period of time.
         * @return
         */
        public float getFitness() 
        {
            return (float)( totalDamageCaused + totalDamageCaused * ( totalTimeLived / Instinct.MAX_TIME_TO_LIVE ) );
        }


        /** 
         *  Reports damaged caused by a bot with this brain
         *  after the bot was destroyed.
         * @param timeLived 
         */
        public void report(long damageCaused, long timeLived) 
        {
            totalDamageCaused = damageCaused;
            totalTimeLived = timeLived;
        }


        /**
            Returns a smaller number if this brain caused more
            damage that the specified object, which should
            also be a brain.
        */
        public int compareTo(Object object) 
        {
            BrainStats other = (BrainStats)object;
            float thisScore = this.getFitness();
            float otherScore = other.getFitness();
            if (thisScore == 0 && otherScore == 0) 
            {
                return 0;
            }
            else 
            {
                // higher fitness is better
                return (int)MoreMath.sign( otherScore - thisScore );
            }
        }

        /**
         * Initial creation of the brain
         */
		public void create() 
		{
			chanceToRun = MoreMath.random(100f);
			mutationRun = MoreMath.random(100f);
			
		    chanceToCharge = MoreMath.random(100f);
	        mutationCharge = MoreMath.random(100f);
	        
		    chanceToRangedAttack = MoreMath.random(100f);
	        mutationRanged = MoreMath.random(100f);
	        
		    chanceToLayTrap = MoreMath.random(100f);
	        mutationTrap = MoreMath.random(100f);
	        
		    chanceToEnchant = MoreMath.random(100f);
	        mutationEnchant = MoreMath.random(100f);
	        
	        fixProbabilites();
		}


        /**
            Mutates this brain. The specified mutationProbability
            is the probability that each brain attribute
            becomes a different value, or "mutates".
        */
        public void mutate() 
        {
        	// mutate the values using the step sizes
        	chanceToRun = chanceToRun + mutationRun * MoreMath.gausian();
			chanceToCharge = chanceToCharge + mutationCharge * MoreMath.gausian();
	        chanceToRangedAttack = chanceToRangedAttack + mutationRanged * MoreMath.gausian();
	        chanceToLayTrap = chanceToLayTrap + mutationTrap * MoreMath.gausian();
	        chanceToEnchant = chanceToEnchant + mutationEnchant * MoreMath.gausian();
		    
	        // mutate the step sizes using the learning rate
			mutationRun = mutationRun * Math.pow( Math.E, ( CHANGE_IN_LEARNING * RATE_OF_LEARNING * MoreMath.gausian() ) );
	        mutationCharge = mutationCharge * Math.pow( Math.E, ( CHANGE_IN_LEARNING * RATE_OF_LEARNING * MoreMath.gausian() ) );
	        mutationRanged = mutationRanged * Math.pow( Math.E, ( CHANGE_IN_LEARNING * RATE_OF_LEARNING * MoreMath.gausian() ) );
	        mutationTrap = mutationTrap * Math.pow( Math.E, ( CHANGE_IN_LEARNING * RATE_OF_LEARNING * MoreMath.gausian() ) );
	        mutationEnchant = mutationEnchant * Math.pow( Math.E, ( CHANGE_IN_LEARNING * RATE_OF_LEARNING * MoreMath.gausian() ) );
	        
            fixProbabilites();
        }


        public Object clone() {
            BrainStats brain = new BrainStats(this);
            brain.generation++;
            return brain;
        }

        public String toString() 
        {
                return "Fitness for bot: " + getFitness() +
                    "Generation: " + generation + "\n" +
                    super.toString();
        }
    }

    private ArrayList< BrainStats > brains;

    public EvolutionGenePool() 
    {
        // make a few random brains to start
        brains = new ArrayList< BrainStats >();
        for (int i=0; i<NUMBER_OF_TOTAL_BRAINS; i++) {
            BrainStats brain = new BrainStats();
            // randomize (mutate) all brain properties
            brain.create();
            brains.add(brain);
        }
    }


    public void resetEvolution() {
        brains.clear();
    }

    private ArrayList< BrainStats > parents = new ArrayList< BrainStats >();
    
    /**
     * This function is called periodically to see if all the children have been
     * evaluated yet.  If they have been, then the best of them are chosen as parents
     * for the next generation, with ALL from the last generation being lost.
     */
	public synchronized void update() 
	{
		if( brains.size() == NUMBER_OF_TOTAL_BRAINS )
		{
			parents.clear();
			
			for( int i = 0; i < NUMBER_OF_TOP_BRAINS; i++ )
			{
				parents.add( brains.get( i ) );
			}
			
			for( BrainStats brain:parents )
			{
				brain.mutate();
			}
			
			brains.clear();
			
			// Creates the children
			for( int i = 0; i < NUMBER_OF_TOTAL_BRAINS; i++ )
			{
				brains.add( crossover() );
			}
		}
	}


	private BrainStats brain;
	
    private BrainStats crossover() 
    {
    	brain = new BrainStats();
    	
    	// This is getting a random parent from the list of parents and taking their stat for a
    	// given stat.
    	brain.chanceToCharge = gettingBrainStats( parents ).chanceToCharge;
    	brain.chanceToEnchant = gettingBrainStats( parents ).chanceToEnchant;
    	brain.chanceToLayTrap = gettingBrainStats( parents ).chanceToLayTrap;
    	brain.chanceToRangedAttack = gettingBrainStats( parents ).chanceToRangedAttack;
    	brain.chanceToRun = gettingBrainStats( parents ).chanceToRun;
    	
    	averageEachStatFromOneOrMoreParents( brain, parents );
 
		return brain;
	}
    
    /**
     * This forces the object from the list into a specific type, so methods and values can be drawn
     * straight from it.  For some odd reason, the compiler wouldn't let me explicitly define it's 
     * type, so I made this.
     * @param list
     * @return
     */
    private BrainStats gettingBrainStats( List list )
    {
    	return (BrainStats) MoreMath.random( list );
    }
    
    /**
     * this is only for use with crossover()  The way it is set up, the same parent could be
     * chosen twice, thus the stat is copied over to the child, or two different parents are
     * chosen, and the stat from each is averaged and given to the child.
     */
    private void averageEachStatFromOneOrMoreParents( BrainStats brain, ArrayList<BrainStats> parents)
    {
    	brain.mutationCharge = average( gettingBrainStats( parents ).mutationCharge, 
    			gettingBrainStats( parents ).mutationCharge );
		brain.mutationEnchant = average( gettingBrainStats( parents ).mutationEnchant,
				 gettingBrainStats( parents ).mutationEnchant );
		brain.mutationRanged = average( gettingBrainStats( parents ).mutationRanged,
				 gettingBrainStats( parents ).mutationRanged );
		brain.mutationRun = average( gettingBrainStats( parents ).mutationRun,
				 gettingBrainStats( parents ).mutationRun );
		brain.mutationTrap = average( gettingBrainStats( parents ).mutationTrap,
				 gettingBrainStats( parents ).mutationTrap);
    }
    
    private double average( double brain1, double brain2 )
    {
    	return ( brain1 + brain2 )/ 2;
    }


	/**
        Gets a new brain from the gene pool. The brain will either
        be a "top" brain or a new, mutated "top" brain.
    */
    public Instinct getNewBrain() 
    {
    	for( Instinct brain:brains )
    	{
    		if( brain.alive )
    		{
    			continue;
    		}
    		else
    		{
    			brain.alive = true;
    			return brain;
    		}
    	}
    	
    	// This statement should only be reached if there is no more 
    	// un-evaluated brains in the list.
    	update();
		return getNewBrain();
    }


    /**
        Notify that a creature with the specified brain has
        been destroyed. The brain's stats are recorded. If the
        brain's stats are within the top
    */
    public void notifyDead(Instinct brain, long damageCaused, long timeLived) {
        // update statistics for this brain
        if ( brain instanceof BrainStats ) 
        {
            BrainStats stat = ( BrainStats ) brain;

            // report the damage
            stat.report( damageCaused, timeLived );

            // sort and trim the list
            if ( !brains.contains( stat ) ) 
            {
                brains.add( stat );
            }
            
            // This sorts the brains, from highest to lowest value.
            // Since it doesn't actually count the fitness values, only whether
            // one is higher than another or not, it doesn't find the maximum
            // too quickly, nor is it as likely to get stuck on local maxima.
            Collections.sort(brains);
            while (brains.size() > NUMBER_OF_TOTAL_BRAINS) {
                brains.remove(NUMBER_OF_TOTAL_BRAINS);
            }
        }
    }
    
    public String toString() {

        // display best brains
        String retVal = "Top " + NUMBER_OF_TOP_BRAINS + " Brains:\n";
        for (int i=0; i<NUMBER_OF_TOP_BRAINS; i++) {
            retVal+= (i+1) + ".\n";
            retVal+=brains.get(i) + "\n";
        }

        return retVal;
    }
}
