package objects.Spells;
// LaserShot.java
// Andrew Davison, May 2003, dandrew@ratree.psu.ac.th

/*  Creates the scene subgraph:

      beamTG --> beamSW --> beamDir TG --> beam cylinder
                       |
                        --> explShape (explosion anim)
     
    The beamTG is for moving the beam and explosion. The Switch is
    for visibility (nothing, beam, or explosion), and the beamDir TG
    is for tweaking the position of the cylinder and rotating it.

    The KeyBehavior object calls fireBeam() in AmmoManager, which
    then calls requestFiring() in each of its LaserShot objects. 

    If the LaserShot is not in use (inUse == false) then LaserShot starts
    an AnimBeam thread which calls LaserShot's moveBeam(). inUse is set
    to true.

    moveBeam() incrementally moves the beam forward, starting from the 
    current viewer position (steerTG). If the beam gets close
    to the target then the explosion is shown, otherwise the beam disappears
    after reaching a ceratin MAX_RANGE from the gun. inUse is set to false 
    again.
*/

import java.text.DecimalFormat;
import java.util.ArrayList;

import javax.media.j3d.*;
import com.sun.j3d.utils.geometry.*;
import javax.vecmath.*;

import objects.DynamicObject;
import objects.DynamicObjectManager;
import objects.Effects.AnimationBeam;
import objects.Effects.Explosion;
import utils.Logger;

import main.TerrainManager;


public abstract class Spell
{
  private static final double MAX_RANGE = 10.0;  // max distance traveled
  private static final double STEP = 0.10;       // distance traveled each increment
  private static final Vector3d INCRIMENTAL_VECTOR = new Vector3d(0, 0, -STEP);
  private static final long SLEEP_TIME = 10;     // delay between moves of spell
  private static final float RADIUS = 0.05f;     // Radius of the sphere
private static final float DETECT_RADIUS = 0.25f; // Radius of the collision detection sphere
public static final char RANGED = 'r';
public static final char TRAP = 't';
public static final char ENCHANT = 'e';
public static final String FIRE = "fire";
private static final int DAMAGE = 20;
private int damageDone = 0;
private String type;

  private TransformGroup steerTransformGroup;   // the TG for the user's viewpoint

  private TransformGroup beamTransfomrGroup;    // moving the beam/explosion
  private Switch beamSwitch;            // for beam/explosion visibility
  private Sphere beam;            // the beam is a sphere
  private Explosion explosion;    // used to create the explosion object(s)
  private TerrainManager terrain;   // For collision detection

  private boolean inUse;        // whether beam is in flight or not

  // for repeated calculations
  private Transform3D temporaryTtransform3d = new Transform3D();
  private Transform3D toMove = new Transform3D();
  private Transform3D localTransform3d = new Transform3D();
  private Vector3d currentVector = new Vector3d();
  private Logger log;
  private DynamicObjectManager objectManager;

  public Spell(TransformGroup steerTransformGroup,
				String type, TerrainManager terrain, Logger log, DynamicObjectManager objectManager)
  {
	  this.objectManager = objectManager;
    this.steerTransformGroup = steerTransformGroup;
    makeBeam(type);
    inUse = false;
    this.terrain= terrain;
    this.log = log;
   }  // end of Spell()


  private void makeBeam(String type)
  // Create the beam and explosion subgraph detailed above.
  {
    // beam position
    Transform3D beamPositionTransform3d = new Transform3D();
    beamPositionTransform3d.rotX(Math.PI/2);   // 90 degrees, so pointing into scene
    beamPositionTransform3d.setTranslation(new Vector3d(0, -0.3, -0.70));  
    // down and outside the object firing (the 0.70 is the closest it could start w/o
    // triggering the collision detection
    TransformGroup beamDirectionGroup = new TransformGroup();
    beamDirectionGroup.setTransform(beamPositionTransform3d);

    beam = new Sphere( RADIUS, makeAppearance() );  // thin red cylinder

    beamDirectionGroup.addChild( beam );

    // create switch for visibility
    beamSwitch = new Switch();
    beamSwitch.setCapability(Switch.ALLOW_SWITCH_WRITE);

    beamSwitch.addChild( beamDirectionGroup );       // add beam to the switch
    beamSwitch.setWhichChild( Switch.CHILD_NONE );   // invisible initially

    // add the explosion to the switch, centered at (0,0,0), size 1.0f
    explosion = new Explosion( 0.50f, type, objectManager);
    beamSwitch.addChild( explosion.getExplosion() );
    beamSwitch.setWhichChild( Switch.CHILD_NONE );   // invisible initially

    // top-level beam TG
    beamTransfomrGroup = new TransformGroup();
    beamTransfomrGroup.addChild( beamSwitch );
    beamTransfomrGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_WRITE);
    beamTransfomrGroup.setCapability(TransformGroup.ALLOW_TRANSFORM_READ);
  } // end of makeBeam()

/**
 * Need to overwrite this method, since it creates a blank Appearance
 * @return
 */
  protected abstract Appearance makeAppearance();


  public TransformGroup getTransformGroup()
  // accessed by AmmoManager
  {  return beamTransfomrGroup;  }



  // ------------------------ shot methods ----------------


  public boolean requestFiring(char mode)
  // If the beam is not being used then fire it
  { if (inUse)
    {
      return false;
    }
    else 
    {
      inUse = true;
      switch( mode )
      {
      case RANGED:
    	  new AnimationBeam(this).start(); // calls moveBeam() inside a thread
    	  break;
      case TRAP:
    	  terrain.addDynamicObject(this);
    	  log.println("create a trap here");
    	  break;
      case ENCHANT:
    	  log.println("enchant the being casting this");
    	  break;
      default:
    	  return false;
      }
      return true;
    }
  }  // end of fireBeam()


  public void moveBeam()
  /* Move the beam with a delay between each move.
     If the beam gets close to the target then show the explosion
     otherwise the beam disappears when it reaches MAX_RANGE
  */
  { // place the beam at the current gun hand position
    steerTransformGroup.getTransform( temporaryTtransform3d );
    beamTransfomrGroup.setTransform( temporaryTtransform3d);
    showBeam(true);
    
    beamTransfomrGroup.getTransform( localTransform3d );    // get beam's TG in world coordinates
    localTransform3d.get(currentVector);

    double currentDistance = 0.0;
    boolean hitObsticle = !terrain.canMoveTo(currentVector); // If it can't move, then it has hit an obstacle
    DynamicObject target = attackCollision( currentVector, DETECT_RADIUS );
    boolean hitTarget = false;
    if( target != null )
    {
    	hitTarget = true;
    }
    while ((currentDistance < MAX_RANGE) && !hitObsticle && !hitTarget) 
    {
      doMove(INCRIMENTAL_VECTOR);
      beamTransfomrGroup.getTransform( localTransform3d );    // get beam's TG in world coordinates
      localTransform3d.get(currentVector);
      log.println(currentVector.toString());
      hitObsticle = !terrain.canMoveTo(currentVector);
      target = attackCollision( currentVector, DETECT_RADIUS );
      if( target != null )
      {
      	hitTarget = true;
      	break;
      }
      
      currentDistance += STEP;
      try {
        Thread.sleep(SLEEP_TIME);    
      } 
      catch (Exception e) {}
    }
    log.println( " inside laser shot impact "+currentDistance+" hit target:"+hitObsticle );

    showBeam( false );     // make beam invisible
    if( hitObsticle )
    {
      showExplosion();   // if a hit, show explosion
    }
    else if( hitTarget )
    {
    	collidingEffectOn( target );
    	showExplosion();
    }
    inUse = false;       // shot is finished
  }  // end of moveBeam()
  

    private ArrayList< DynamicObject > objectList;
	private DynamicObject attackCollision( Vector3d currentVector2,
		float detectRadius ) 
	{
		if ( objectManager != null ) 
		{
			objectList = objectManager.getObjectList();
			if (objectList != null && objectList.size() > 0) {
				for (DynamicObject object : objectList) 
				{
					if (object.getRange() > objectManager.range(currentVector2,
							object.getPosition())
							|| DETECT_RADIUS > objectManager.range(
									currentVector2, object.getPosition())) 
					{
						return object;
					}
				}
			}
		}
		return null;
	}


	public void collidingEffectOn(DynamicObject data) 
	{
		int typeIntCast = 0;
		
		if( type.equalsIgnoreCase( Spell.FIRE ) )
		{
			typeIntCast = 1;
		}
		else
		{
			// start of if-else if-else line, once there are more types of spells
		}
		
		damageDone += data.doDamage( new int[] { DAMAGE, typeIntCast } );
	}


  private void doMove(Vector3d moveVector)
  {
    beamTransfomrGroup.getTransform( temporaryTtransform3d );
    toMove.setTranslation( moveVector );
    temporaryTtransform3d.mul(toMove);
    beamTransfomrGroup.setTransform( temporaryTtransform3d );
  } // end of doMove()


  private void showBeam(boolean toVisible)
  // make beam visible/invisible
  {
    if (toVisible)
      beamSwitch.setWhichChild(0);   // make visible
    else
      beamSwitch.setWhichChild( Switch.CHILD_NONE );   // make invisible
  }  // end of showBeam()


  private void showExplosion()
  // start the explosion
  {
    beamSwitch.setWhichChild(1);   // make visible
    explosion.showSeries();    // boom!
    beamSwitch.setWhichChild( Switch.CHILD_NONE );   // make invisible
  }  // end of showExplosion()



  // -------------------- for debugging ---------------------


  private void printTransformGroup(TransformGroup transformGroup, String name)
  // display info about TransformGroup called name
  {
     Transform3D transform3d = new Transform3D();
     transformGroup.getTransform( transform3d );
     // log.println(name + " Transform:\n" + t3d);
     Vector3d vec = new Vector3d();
     transform3d.get(vec);

     DecimalFormat dedimalFormat = new DecimalFormat("0.00");  // 2 dp
     // log.println(name + " Vector: " + vec);
     log.println(name + " Vector: (" +
			dedimalFormat.format(vec.x) + ", " + dedimalFormat.format(vec.y) + ", " +
			dedimalFormat.format(vec.z) + ")");
  }  // end of printTG()


}  // end of LaserShot class
