﻿using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.Xna.Framework;
using Microsoft.Xna.Framework.Audio;
using Microsoft.Xna.Framework.Content;
using Microsoft.Xna.Framework.GamerServices;
using Microsoft.Xna.Framework.Graphics;
using Microsoft.Xna.Framework.Input;
using Microsoft.Xna.Framework.Media;
using Microsoft.Xna.Framework.Net;
using Microsoft.Xna.Framework.Storage;
using XSIXNARuntime;

namespace Hook_FPS
{
   /// <summary>
    /// This class store all the information and statistics about a player,
    /// its model asset, its camera asset, and its bullets.
    /// </summary>
   class Player
   {
        #region Private Fields

        // Model, Camera, Bullets, BoundingSpheres
        private ModelAsset _model;
        private CameraAsset _camera;
        private BulletObject[] _bullets;
        private BoundingSphere[] _boundingSpheres;

        // Keyboard, Mouse, and Animation variables
        private KeyboardState _keyboardState;
        private MouseState _originalMouseState;
        private MouseState _previousMouseState;
        
        // Movement Variables
        private TimeSpan _jumpTimeSpan;
        private float _velocity; // keeps track of the current forward velocity (Range: 0 to 5)
        private float _upVelocity; // keeps track of the current upward velocity (Range: -5 to 5)
        private bool _isJumping; // keeps track of the jump animation status
        private bool _gravityChanged; // true if a gravity orientation just took place.
        
        // Constants: Animations, Gravity, Speed
        private const int TPOSE = 0;
        private const int STAND = 1;
        private const int RUN = 2;
        private const int JUMP = 3;
        private const int DIE = 4;
        private const int RUNBACKWARDS = 5;

        // Constants: Acceleration, Gravity, Move speed
        private const float GRAVITY_ACCELERATION = -0.1f;
        private const float ACCELERATION = 0.25f;
        private const float MAXSPEED = 6.0f;

        // Constants: Bullets
        private const int NUMBULLETS = 30;
        private const float BULLETSPEED = 50.0f;
        
        // Constants: BoundSphere Indices
        private enum spheres
        {
            BIG, BODY, UP, DOWN, LEFT, RIGHT, FRONT, BACK, GROUND, GROUNDFRONT, GROUNDBACK, GROUNDLEFT, GROUNDRIGHT, HEAD, SIZE
        }
            
        // Player Stats
        private int _playerNumber;
        private bool _isAlive = true;
        private int _health = 100;
        private int _weaponState = 0;
        private TimeSpan _coolDown;
        private const int OVERHEAT = 41;
        private const int OVERHEATWAIT = 4;

        // TODO: add aditional stats (Health, Kills, ect)

        // Map Bounding Boxes
        private BoundingBox[] _mapWalls;

        #endregion

        #region Properties

        public ModelAsset Model { get { return _model; } set { _model = value; } }
        public CameraAsset Camera { get { return _camera; } set { _camera = value; } }
        public bool IsAlive { get { return _isAlive; } set { _isAlive = value; } }
        public BulletObject[] Bullets { get { return _bullets; } }
        public BoundingSphere[] BoundSpheres { get { return _boundingSpheres; } }
        public int Health { get { return _health; } set { _health = value; } }
        public int WeaponState { get { return _weaponState; } set { _weaponState = value; } }
        public int PlayerNumber { get{return _playerNumber;} set{_playerNumber = value;} }
        
        #endregion

        #region Constructor(s)

        /// <summary>
        /// Constructs a new Player object.
        /// </summary>
        /// <param name="ModelAssetPath">The directory path for the model</param>
        /// <param name="CameraAssetPath">The directory path for the camera and scene</param>
        /// <param name="content">The Game content manager</param>
        /// <param name="graphics">The Graphics Device Manager for the player</param>
        /// <param name="newMouseState">The players inital mouse state</param>
        public Player(
            String ModelAssetPath, 
            String CameraAssetPath, 
            ContentManager content, 
            GraphicsDeviceManager graphics, 
            MouseState newMouseState, 
            BoundingBox[] mapBoundingBoxes,
            int playerNum)
        {
            // set player number
            _playerNumber = playerNum;
            
            // Init Model
            _model = new ModelAsset(ModelAssetPath, content);
            _model.Position.Y = 100.0f; // set the initial model position 100 units up so the model doesn't fall below the ground;
            
            // Init Bounding Spheres
            _boundingSpheres = new BoundingSphere[(int)spheres.SIZE];
            BoundingSphere bigSphere = new BoundingSphere(_model.Position, 37.0f); // envelops the entire model
            BoundingSphere bodySphere = new BoundingSphere(_model.Position, 15.0f); // envelops just the core of the body
            BoundingSphere upSphere = new BoundingSphere(_model.Position, 10.0f);
            BoundingSphere downSphere = new BoundingSphere(_model.Position, 10.0f);
            BoundingSphere leftSphere = new BoundingSphere(_model.Position, 10.0f);
            BoundingSphere rightSphere = new BoundingSphere(_model.Position, 10.0f);
            BoundingSphere frontSphere = new BoundingSphere(_model.Position, 10.0f);
            BoundingSphere backSphere = new BoundingSphere(_model.Position, 10.0f);
            BoundingSphere groundSphere = new BoundingSphere(_model.Position, 10.0f);
            BoundingSphere groundFrontSphere = new BoundingSphere(_model.Position, 10.0f);
            BoundingSphere groundBackSphere = new BoundingSphere(_model.Position, 10.0f);
            BoundingSphere groundLeftSphere = new BoundingSphere(_model.Position, 10.0f);
            BoundingSphere groundRightSphere = new BoundingSphere(_model.Position, 10.0f);
            BoundingSphere headSphere = new BoundingSphere(_model.Position, 4.0f);
            _boundingSpheres[(int)spheres.BIG] = bigSphere;
            _boundingSpheres[(int)spheres.BODY] = bodySphere;
            _boundingSpheres[(int)spheres.UP] = upSphere;
            _boundingSpheres[(int)spheres.DOWN] = downSphere;
            _boundingSpheres[(int)spheres.LEFT] = leftSphere;
            _boundingSpheres[(int)spheres.RIGHT] = rightSphere;
            _boundingSpheres[(int)spheres.FRONT] = frontSphere;
            _boundingSpheres[(int)spheres.BACK] = backSphere;
            _boundingSpheres[(int)spheres.GROUND] = groundSphere;
            _boundingSpheres[(int)spheres.GROUNDFRONT] = groundFrontSphere;
            _boundingSpheres[(int)spheres.GROUNDBACK] = groundBackSphere;
            _boundingSpheres[(int)spheres.GROUNDLEFT] = groundLeftSphere;
            _boundingSpheres[(int)spheres.GROUNDRIGHT] = groundRightSphere;
            _boundingSpheres[(int)spheres.HEAD] = headSphere;

            // Init Camera
            _camera = new CameraAsset(CameraAssetPath, content);
            
            // Init Bullets
            _bullets = new BulletObject[NUMBULLETS];
            _bullets[0] = new BulletObject();
            _bullets[0].Model = content.Load<Model>("Content/Models/redLaser");
            for (int i = 1; i < NUMBULLETS; i++)
            {
                _bullets[i] = new BulletObject();
                _bullets[i].Model = _bullets[0].Model;
            }

            // Init Map Walls
            _mapWalls = mapBoundingBoxes;

            // Set the original mouse state
            _originalMouseState = newMouseState;
            _previousMouseState = newMouseState;

            // Init velocities
            _velocity = 0.0f;
            _upVelocity = 0.0f;
        }

        #endregion

        #region Private Methods

        /// <summary>
        /// Checks if a key is pressed.
        /// </summary>
        /// <param name="inKeys">An array of keys from the Keyboard State</param>
        /// <param name="whichKey">The Key to check if pressed</param>
        /// <returns>True if the specified key is pressed; false otherwise</returns>
        private bool CheckKeys(Keys[] inKeys, Keys whichKey)
        {
            // 
            for (int key = 0; key < inKeys.Length; key++)
            {
                if (inKeys[key] == whichKey)
                {
                    return true; // whichKey is pressed
                }
            }
            return false; // whichKey not pressed
        }

        /// <summary>
        /// Update the model animation sequence, rotation, and translation in that order.
        /// </summary>
        /// <param name="gameTime">The current Game Time</param>
        private void UpdateKeyboard(GameTime gameTime, KeyboardState newKeyboardState)
        {
            // Temp varaible for calculating movement intially set to the current forward vector
            Vector3 movement = Vector3.Zero;

            // Get the list of collisions
            List<int> collisions = DetectCollisions();

            // set local booleans for wall collisions
            bool onTheGround = GroundDetected(collisions);
            bool upCollsion = WallUpDetected(collisions);
            bool downCollision = WallDownDetected(collisions);
            bool frontCollision = WallFrontDetected(collisions);
            bool backCollision = WallBackDetected(collisions);
            bool leftCollision = WallLeftDetected(collisions);
            bool rightCollision = WallRightDetected(collisions);
            bool bodyCollision = WallBodyDetected(collisions);
            bool frontGround = GroundFrontDetected(collisions);
            bool backGround = GroundBackDetected(collisions);
            bool leftGround = GroundLeftDetected(collisions);
            bool rightGround = GroundRightDetected(collisions);

            // Was the space key lifted
            bool SpaceLifted = (newKeyboardState.IsKeyDown(Keys.Space) && _keyboardState.IsKeyUp(Keys.Space));

            // Set the new Keyboard state
            _keyboardState = newKeyboardState;

            // Check which keys are pressed one time
            Keys[] keys = _keyboardState.GetPressedKeys();
            bool Wkey = CheckKeys(keys, Keys.W);
            bool Skey = CheckKeys(keys, Keys.S);
            bool Akey = CheckKeys(keys, Keys.A);
            bool Dkey = CheckKeys(keys, Keys.D);
            bool Qkey = CheckKeys(keys, Keys.Q);
            bool Ekey = CheckKeys(keys, Keys.E);
            bool Spacekey = CheckKeys(keys, Keys.Space);
      
            // calculate the elasped time spent jumping
            double jumpTime = gameTime.TotalGameTime.Subtract(_jumpTimeSpan).TotalSeconds;

            // Global boolean
            _isJumping = (_model.AnimationIndex == JUMP);

            // Determine which animation to play
            if (!_isJumping && Spacekey && SpaceLifted && !upCollsion) // Start Jump
            {
                _jumpTimeSpan = gameTime.TotalGameTime;
                _upVelocity += 2.0f;
                _model.OldAnimationIndex = _model.AnimationIndex;
                _model.AnimationIndex = JUMP;
                _isJumping = true;
                _model.Play(gameTime);               
            }
            else if (_isJumping) // Continue Jumping
            {
                if (jumpTime > 0.25 && onTheGround && !upCollsion) // Jump lasted longer than a quarter of a second
                {
                    _isJumping = false;

                    // Set the animation to use since no longer jumping
                    if (Wkey || Akey || Dkey)
                    {
                        _model.OldAnimationIndex = _model.AnimationIndex;
                        _model.AnimationIndex = RUN;
                    }
                    else
                    {
                        _model.OldAnimationIndex = _model.AnimationIndex;
                        _model.AnimationIndex = RUNBACKWARDS;
                    }
                }
            }
            else if ((Wkey || Akey || Dkey) && (!_isJumping) && (!Skey)) // Run forwards
            {
                _model.OldAnimationIndex = _model.AnimationIndex;
                _model.AnimationIndex = RUN;
                if (!_model.PlaybackStatus)
                {
                    _model.Play(gameTime);
                }
            }
            else if ((Skey) && (!_isJumping) && (!Wkey)) // Run backwards
            {
                _model.OldAnimationIndex = _model.AnimationIndex;
                _model.AnimationIndex = RUNBACKWARDS;
                if (!_model.PlaybackStatus)
                {
                    
                    _model.Play(gameTime);
                }
            }
            else // Stand still
            {
                _isJumping = false;
                _model.PlaybackStatus = false;
                _model.OldAnimationIndex = _model.AnimationIndex;
                _model.AnimationIndex = STAND;
                _velocity = 0.0f;
            }

            // Accelerate the forward velocity
            if (Wkey || Skey || Akey || Dkey)
            {
                _velocity += ACCELERATION;
            }

            // Accelerate the upward velocity and slow forward velocity
            if (Spacekey && !upCollsion && jumpTime < 0.25)
            {
                _upVelocity += ACCELERATION;
                _velocity -= ACCELERATION * 1.2f;
                _velocity = MathHelper.Clamp(_velocity, 3.0f, MAXSPEED);
            }
            else
            {
                if (onTheGround)
                {
                    _upVelocity = 0.0f;
                    _gravityChanged = false;
                }
                else // still falling down so apply gravity
                {
                    _upVelocity += GRAVITY_ACCELERATION;
                    _velocity -= ACCELERATION * 1.2f;
                    _velocity = MathHelper.Clamp(_velocity, 3.0f, MAXSPEED);
                }
            }

            if (upCollsion && _upVelocity > 0.0f) // ceiling detected and still moving upwards
            {
                _upVelocity = 0.0f;
            }
            
            _upVelocity = MathHelper.Clamp(_upVelocity, -MAXSPEED, MAXSPEED);
            movement += _model.Up * _upVelocity;

            // Clamp the velocity
            _velocity = MathHelper.Clamp(_velocity, -MAXSPEED, MAXSPEED);
            if (newKeyboardState.IsKeyDown(Keys.LeftShift))
            {
                _velocity = MathHelper.Clamp(_velocity, -(MAXSPEED/3), MAXSPEED/3);
            }

            // Init a temp varaible to use for the model translations
            float velocityToUse = _velocity;
            
            // Adjust the forward velocity for diagonal movement
            if ((Wkey || Skey) && (Akey || Dkey))
            {
                velocityToUse *= 0.6f; // 75% of the current velocity 
            }

            // Move Forward
            if (Wkey && !frontCollision && !frontGround)
            {
                movement -= _model.Forward * velocityToUse;
            }            

            // Move Backward
            if (Skey && !backCollision && !backGround)
            {
                movement += _model.Forward * velocityToUse;
            }

            // Move Left
            if (Akey && !leftCollision && !leftGround)
            {
                movement -= _model.World.Left * velocityToUse;
            }

            // Move Right
            if (Dkey && !rightCollision && !rightGround)
            {
                movement -= _model.World.Right * velocityToUse;
            }
            
            // Hook Right
            if (Ekey && !rightCollision && !rightGround && _isJumping && !_gravityChanged)
            {
                _gravityChanged = true;
                _model.Up = GetNewUpVectorRight();
            }

            // Hook Left
            if (Qkey && !leftCollision && !leftGround && _isJumping && !_gravityChanged)
            {
                _gravityChanged = true;
                _model.Up = GetNewUpVectorLeft();
            }
            
            // One last animation check
            // if the movement vector is (0,0,0) and not jumping
            // make the model stand still
            // movement will be (0,0,0) if the player
            // runs into a wall.
            if (movement.Equals(Vector3.Zero) && !_isJumping)
            {
                _model.AnimationIndex = STAND;
                _velocity = 0.0f;
            }
            
            
            // Translate the model with the new movement vector
            _model.Translate(movement);
        }
        
        /// <summary>
        /// Rotate the gravity right.
        /// </summary>
        /// <returns></returns>
        private Vector3 GetNewUpVectorRight()
        {
            Vector3 upVector = _model.Up;
            float xComponent = _model.Forward.X;
            float yComponent = _model.Forward.Y;
            float zComponent = _model.Forward.Z;
            float absX = Math.Abs(xComponent);
            float absY = Math.Abs(yComponent);
            float absZ = Math.Abs(zComponent);

            if (absX > absY && absX > absZ)
            {
                if (xComponent > 0)
                {
                    if (upVector == Vector3.Up)
                    {
                        upVector = Vector3.Backward;
                    }
                    else if (upVector == Vector3.Backward)
                    {
                        upVector = Vector3.Down;
                    }
                    else if (upVector == Vector3.Down)
                    {
                        upVector = Vector3.Forward;
                    }
                    else if (upVector == Vector3.Forward)
                    {
                        upVector = Vector3.Up;
                    }
                }
                else
                {
                    if (upVector == Vector3.Up)
                    {
                        upVector = Vector3.Forward;
                    }
                    else if (upVector == Vector3.Forward)
                    {
                        upVector = Vector3.Down;
                    }
                    else if (upVector == Vector3.Down)
                    {
                        upVector = Vector3.Backward;
                    }
                    else if (upVector == Vector3.Backward)
                    {
                        upVector = Vector3.Up;
                    }
                }
            }

            else if (absZ > absX && absZ > absY)
            {
                if (zComponent > 0)
                {
                    if (upVector == Vector3.Up)
                    {
                        upVector = Vector3.Left;
                    }
                    else if (upVector == Vector3.Left)
                    {
                        upVector = Vector3.Down;
                    }
                    else if (upVector == Vector3.Down)
                    {
                        upVector = Vector3.Right;
                    }
                    else if (upVector == Vector3.Right)
                    {
                        upVector = Vector3.Up;
                    }
                }
                else
                {
                    if (upVector == Vector3.Up)
                    {
                        upVector = Vector3.Right;
                    }
                    else if (upVector == Vector3.Right)
                    {
                        upVector = Vector3.Down;
                    }
                    else if (upVector == Vector3.Down)
                    {
                        upVector = Vector3.Left;
                    }
                    else if (upVector == Vector3.Left)
                    {
                        upVector = Vector3.Up;
                    }
                }
            }

            else if (absY > absX && absY > absZ)
            {
                if (yComponent > 0)
                {
                    if (upVector == Vector3.Forward)
                    {
                        upVector = Vector3.Left;
                    }
                    else if (upVector == Vector3.Left)
                    {
                        upVector = Vector3.Backward;
                    }
                    else if (upVector == Vector3.Backward)
                    {
                        upVector = Vector3.Right;
                    }
                    else if (upVector == Vector3.Right)
                    {
                        upVector = Vector3.Forward;
                    }
                }
                else
                {
                    if (upVector == Vector3.Backward)
                    {
                        upVector = Vector3.Left;
                    }
                    else if (upVector == Vector3.Left)
                    {
                        upVector = Vector3.Forward;
                    }
                    else if (upVector == Vector3.Forward)
                    {
                        upVector = Vector3.Right;
                    }
                    else if (upVector == Vector3.Right)
                    {
                        upVector = Vector3.Backward;
                    }
                }

            }
            
            Console.WriteLine(upVector + "                   modelForward " + Model.Forward.ToString());
            return upVector;
        }
        
        /// <summary>
        /// Rotate the gravity left.
        /// </summary>
        /// <returns></returns>
        private Vector3 GetNewUpVectorLeft()
        {
            Vector3 rightVector = GetNewUpVectorRight();
            
            if (rightVector == Vector3.Up)
	        {
                return Vector3.Down;
	        }
	        else if (rightVector == Vector3.Down)
	        {
	            return Vector3.Up;
	        }
	        else if (rightVector == Vector3.Left)
	        {
	            return Vector3.Right;
	        }
	        else if (rightVector == Vector3.Right)
	        {
	            return Vector3.Left;
	        }
	        else if (rightVector == Vector3.Forward)
	        {
	            return Vector3.Backward;
	        }
	        else
	        {
                return Vector3.Forward;
	        }    
        }

        /// <summary>
        /// Translates the player's bounding spheres appropriately
        /// according to the model position.
        /// </summary>
        private void UpdatePlayerSpheres()
        {
            // Init the offsets for the six directional spheres
            Vector3 frontBackOffset = _model.Forward * 20; // 20 units in front/back of the model center
            Vector3 leftRightOffset = _model.World.Left * 20; // 20 units to the left/right of the model center
            Vector3 downOffset = _model.Up * 20; // 23 units down from the model center
            Vector3 upOffset = _model.Up * 20; // 20 units up from the model center
            Vector3 groundOffset = _model.Up * 33; //33 units down from the model center
            Vector3 headOffset = upOffset - _model.Forward * 8;
            
            // Translate the spheres
            _boundingSpheres[(int)spheres.BIG].Center = _model.Position;
            _boundingSpheres[(int)spheres.BODY].Center = _model.Position;
            _boundingSpheres[(int)spheres.UP].Center = _model.Position + upOffset;
            _boundingSpheres[(int)spheres.DOWN].Center = _model.Position - downOffset;
            _boundingSpheres[(int)spheres.LEFT].Center = _model.Position - leftRightOffset;
            _boundingSpheres[(int)spheres.RIGHT].Center = _model.Position + leftRightOffset;
            _boundingSpheres[(int)spheres.FRONT].Center = _model.Position - frontBackOffset;
            _boundingSpheres[(int)spheres.BACK].Center = _model.Position + frontBackOffset;
            _boundingSpheres[(int)spheres.GROUND].Center = _model.Position - groundOffset;
            _boundingSpheres[(int)spheres.GROUNDFRONT].Center = _model.Position - downOffset - frontBackOffset;
            _boundingSpheres[(int)spheres.GROUNDBACK].Center = _model.Position - downOffset + frontBackOffset;
            _boundingSpheres[(int)spheres.GROUNDLEFT].Center = _model.Position - downOffset - leftRightOffset;
            _boundingSpheres[(int)spheres.GROUNDRIGHT].Center = _model.Position - downOffset + leftRightOffset;
            _boundingSpheres[(int)spheres.HEAD].Center = _model.Position + headOffset;
        }

        /// <summary>
        /// Update the model yaw and pitch from mouse input and determine
        /// if the left mouse button was clicked and released.  
        /// </summary>
        /// <param name="newMouseState">The new mouse state from the player</param>
        /// <returns>True if left mouse button was clicked once; false otherwise</returns>
        private bool UpdateMouse(MouseState newMouseState, GameTime gameTime)
        {
            // Mouse sensitivity
            // TODO: Allow this to be adjustable by each player
            float sensitivity = 0.001f;  

            // Mouse rotation
            // TODO; Adjust yaw/pitch/roll for HOOK gravity
            if (newMouseState != _originalMouseState)
            {
                float xDifference = newMouseState.X - _originalMouseState.X;
                float yDifference = newMouseState.Y - _originalMouseState.Y;
                float yaw = -xDifference * sensitivity;
                float pitch = yDifference * sensitivity;
                //_model.RotateAboutY(yaw);
                //_model.RotateAboutX(pitch);
                _model.Yaw = yaw;
                _model.Pitch = pitch;
                _camera.RotateAboutY(yaw);
                _camera.RotateAboutX(pitch);
                
                
            }
            
            // Semi-Automatic Firing
            // HashCode of 1 means the mouse button has been clicked
            // Makes sure the mouse button has been unclicked before
            // allowing another bullet to be shot.
            //if (_coolDown == null)
            //{
             //   _coolDown = gameTime.TotalGameTime;
            //}
            
            if ((newMouseState.LeftButton.GetHashCode() == 1) && gameTime.TotalGameTime.Subtract(_coolDown).TotalMilliseconds > 100)// && (newMouseState.LeftButton.GetHashCode() != _previousMouseState.LeftButton.GetHashCode()))
            {
                _coolDown = gameTime.TotalGameTime;
                _previousMouseState = newMouseState;
                return true; // shoot a bullet
            }
            else
            {
                _previousMouseState = newMouseState;
                return false; // don't shoot
            }
            
        }

        /// <summary>
        /// Shoot a bullet if the player has a bullet to shoot
        /// </summary>
        /// <param name="position">The starting position of the bullet</param>
        /// <param name="forward">The forward vector of the bullet (the direction to travel)</param>
        /// <param name="up">The up vector of the bullet (this will make sure the bullet is correctly oriented)</param>
        private void Shoot(Vector3 position, Vector3 forward, Vector3 up)
        {
            // Find the first dead bullet, fire it, and break the loop
            for (int bulletNum = 0; bulletNum < _bullets.Length; bulletNum++)
            {
                if (!_bullets[bulletNum].IsAlive)
                {
                    _bullets[bulletNum].Shooter = this._playerNumber;
                    _bullets[bulletNum].Position = position;
                    _bullets[bulletNum].Forward = forward;
                    _bullets[bulletNum].Up = up;
                    _bullets[bulletNum].Velocity = _bullets[bulletNum].Forward * BULLETSPEED;
                    _bullets[bulletNum].Position += _bullets[bulletNum].Velocity;
                    _bullets[bulletNum].IsAlive = true;
                    _bullets[bulletNum].BoundSphere.Center = position;
                    _bullets[bulletNum].BoundSphere1.Center = position + forward * 4;
                    _bullets[bulletNum].BoundSphere2.Center = position - forward * 4;
                    _bullets[bulletNum].BoundSphere3.Center = position + forward * 8;
                    _bullets[bulletNum].BoundSphere4.Center = position - forward * 8;
                    _bullets[bulletNum].BoundSphere5.Center = position + forward * 12;
                    _bullets[bulletNum].BoundSphere6.Center = position - forward * 12;
                    break;
                }
            }
        }

        /// <summary>
        /// Updates each live fired bullet.  The bullet
        /// position is updated by adding the bullet velocity
        /// vector.
        /// </summary>
        private void UpdateBullets()
        {
            for (int bulletNum = 0; bulletNum < _bullets.Length; bulletNum++)
            {
                if (_bullets[bulletNum].IsAlive
                    && _mapWalls[_mapWalls.Length-1].Contains(_bullets[bulletNum].BoundSphere).Equals(ContainmentType.Contains))
                {

                    
                    //if (!_boundingSpheres[(int)spheres.BODY].Contains(_bullets[bulletNum].BoundSphere).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.BODY].Contains(_bullets[bulletNum].BoundSphere1).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.BODY].Contains(_bullets[bulletNum].BoundSphere2).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.BODY].Contains(_bullets[bulletNum].BoundSphere3).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.BODY].Contains(_bullets[bulletNum].BoundSphere4).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.BODY].Contains(_bullets[bulletNum].BoundSphere5).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.BODY].Contains(_bullets[bulletNum].BoundSphere6).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.DOWN].Contains(_bullets[bulletNum].BoundSphere).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.DOWN].Contains(_bullets[bulletNum].BoundSphere1).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.DOWN].Contains(_bullets[bulletNum].BoundSphere2).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.DOWN].Contains(_bullets[bulletNum].BoundSphere3).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.DOWN].Contains(_bullets[bulletNum].BoundSphere4).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.DOWN].Contains(_bullets[bulletNum].BoundSphere5).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.DOWN].Contains(_bullets[bulletNum].BoundSphere6).Equals(ContainmentType.Disjoint))
                    //{
                    //    _bullets[bulletNum].IsAlive = false;
                    //    _health -= 10;
                    //}
                    //else if (!_boundingSpheres[(int)spheres.HEAD].Contains(_bullets[bulletNum].BoundSphere).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.HEAD].Contains(_bullets[bulletNum].BoundSphere1).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.HEAD].Contains(_bullets[bulletNum].BoundSphere2).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.HEAD].Contains(_bullets[bulletNum].BoundSphere3).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.HEAD].Contains(_bullets[bulletNum].BoundSphere4).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.HEAD].Contains(_bullets[bulletNum].BoundSphere5).Equals(ContainmentType.Disjoint)
                    //    || !_boundingSpheres[(int)spheres.HEAD].Contains(_bullets[bulletNum].BoundSphere6).Equals(ContainmentType.Disjoint))
                    //{
                    //    _bullets[bulletNum].IsAlive = false;
                    //    _health = 0;
                    //}
                    // update bullet and bounding sphere position
                    _bullets[bulletNum].Position -= _bullets[bulletNum].Velocity;
                    _bullets[bulletNum].BoundSphere.Center = _bullets[bulletNum].Position;
                    _bullets[bulletNum].BoundSphere1.Center = _bullets[bulletNum].Position + _bullets[bulletNum].Forward * 4;
                    _bullets[bulletNum].BoundSphere2.Center = _bullets[bulletNum].Position - _bullets[bulletNum].Forward * 4;
                    _bullets[bulletNum].BoundSphere3.Center = _bullets[bulletNum].Position + _bullets[bulletNum].Forward * 8;
                    _bullets[bulletNum].BoundSphere4.Center = _bullets[bulletNum].Position - _bullets[bulletNum].Forward * 8;
                    _bullets[bulletNum].BoundSphere5.Center = _bullets[bulletNum].Position + _bullets[bulletNum].Forward * 12;
                    _bullets[bulletNum].BoundSphere6.Center = _bullets[bulletNum].Position - _bullets[bulletNum].Forward * 12;
                    for (int wallNum = 0; wallNum < _mapWalls.Length - 1; wallNum++)
                    {
                        if (!_mapWalls[wallNum].Contains(_bullets[bulletNum].BoundSphere).Equals(ContainmentType.Disjoint)
                            || !_mapWalls[wallNum].Contains(_bullets[bulletNum].BoundSphere1).Equals(ContainmentType.Disjoint)
                            || !_mapWalls[wallNum].Contains(_bullets[bulletNum].BoundSphere2).Equals(ContainmentType.Disjoint)
                            || !_mapWalls[wallNum].Contains(_bullets[bulletNum].BoundSphere3).Equals(ContainmentType.Disjoint)
                            || !_mapWalls[wallNum].Contains(_bullets[bulletNum].BoundSphere4).Equals(ContainmentType.Disjoint)
                            || !_mapWalls[wallNum].Contains(_bullets[bulletNum].BoundSphere5).Equals(ContainmentType.Disjoint)
                            || !_mapWalls[wallNum].Contains(_bullets[bulletNum].BoundSphere6).Equals(ContainmentType.Disjoint))
                        {
                            _bullets[bulletNum].IsAlive = false;
                        }
                    }
                }
                else
                {
                    _bullets[bulletNum].IsAlive = false;
                }
            }
        }
        
        #endregion
        
        #region Collision Detection Methods
        
        /// <summary>
        /// Determine whether the player's big enveloping
        /// bounding sphere intersects with one of the map bouding
        /// boxes.
        /// </summary>
        /// <returns>A list of the walls the big sphere intersects</returns>
        private List<int> DetectCollisions()
        {
            List<int> walls = new List<int>();
            
            for (int wallNum = 0; wallNum < _mapWalls.Length-1; wallNum++)
            {
                // See if there is a collision with the Big enveloping sphere first
                if (_boundingSpheres[(int)spheres.BIG].Intersects(_mapWalls[wallNum]))
                {
                    if (_boundingSpheres[(int)spheres.GROUND].Intersects(_mapWalls[wallNum]))
                    {
                        walls.Add((int)spheres.GROUND);
                    }
                    if (_boundingSpheres[(int)spheres.GROUNDFRONT].Intersects(_mapWalls[wallNum]))
                    {
                        walls.Add((int)spheres.GROUNDFRONT);
                    }
                    if (_boundingSpheres[(int)spheres.GROUNDBACK].Intersects(_mapWalls[wallNum]))
                    {
                        walls.Add((int)spheres.GROUNDBACK);
                    }
                    if (_boundingSpheres[(int)spheres.GROUNDRIGHT].Intersects(_mapWalls[wallNum]))
                    {
                        walls.Add((int)spheres.GROUNDRIGHT);
                    }
                    if (_boundingSpheres[(int)spheres.GROUNDLEFT].Intersects(_mapWalls[wallNum]))
                    {
                        walls.Add((int)spheres.GROUNDLEFT);
                    }
                    if (_boundingSpheres[(int)spheres.BODY].Intersects(_mapWalls[wallNum]))
                    {
                        walls.Add((int)spheres.BODY);
                    }
                    if (_boundingSpheres[(int)spheres.UP].Intersects(_mapWalls[wallNum]))
                    {
                        walls.Add((int)spheres.UP);
                    }
                    if (_boundingSpheres[(int)spheres.DOWN].Intersects(_mapWalls[wallNum]))
                    {
                        walls.Add((int)spheres.DOWN);
                    }
                    if (_boundingSpheres[(int)spheres.FRONT].Intersects(_mapWalls[wallNum]))
                    {
                        walls.Add((int)spheres.FRONT);
                    }
                    if (_boundingSpheres[(int)spheres.BACK].Intersects(_mapWalls[wallNum]))
                    {
                        walls.Add((int)spheres.BACK);
                    }
                    if (_boundingSpheres[(int)spheres.LEFT].Intersects(_mapWalls[wallNum]))
                    {
                        walls.Add((int)spheres.LEFT);
                    }
                    if (_boundingSpheres[(int)spheres.RIGHT].Intersects(_mapWalls[wallNum]))
                    {
                        walls.Add((int)spheres.RIGHT);
                    }
                }
            }
            
            return walls;
        }

        /// <summary>
        /// Detects if the ground bounding sphere has intersected with a wall.
        /// </summary>
        /// <returns>True if and intersection is found; false otherwise</returns>
        private bool GroundDetected(List<int> collisions)
        {
            // No collisions in the detection list
            if (collisions.Count == 0)
            {
                return false;
            }
            else
            {
                foreach (int sphere in collisions)
                {
                    if (sphere == (int)spheres.GROUND)
                    {
                        return true; // collision Detected
                    }
                }
            }

            return false; // No  collision
        }

        /// <summary>
        /// Detects a collison with the left sphere
        /// </summary>
        /// <param name="collisions"></param>
        /// <returns></returns>
        private bool WallLeftDetected(List<int> collisions)
        {
           // No collisions in the detection list
           if (collisions.Count == 0)
           {
              return false;
           }
           else
           {
              foreach (int sphere in collisions)
              {
                 if (sphere == (int)spheres.LEFT)
                 {
                    return true; //  collision Detected
                 }
              }
           }

           return false; // No  collision
        }

        /// <summary>
        /// Detects a collision with the right sphere
        /// </summary>
        /// <param name="collisions"></param>
        /// <returns></returns>
        private bool WallRightDetected(List<int> collisions)
        {
           // No collisions in the detection list
           if (collisions.Count == 0)
           {
              return false;
           }
           else
           {
              foreach (int sphere in collisions)
              {
                 if (sphere == (int)spheres.RIGHT)
                 {
                    return true; //  collision Detected
                 }
              }
           }

           return false; // No collision
        }

        /// <summary>
        /// Detects a collision with the front sphere
        /// </summary>
        /// <param name="collisions"></param>
        /// <returns></returns>
        private bool WallFrontDetected(List<int> collisions)
        {
           // No collisions in the detection list
           if (collisions.Count == 0)
           {
              return false;
           }
           else
           {
              foreach (int sphere in collisions)
              {
                 if (sphere == (int)spheres.FRONT)
                 {
                    return true; // collision Detected
                 }
              }
           }

           return false; // No collision
        }

        /// <summary>
        /// Detects a collision with the back sphere
        /// </summary>
        /// <param name="collisions"></param>
        /// <returns></returns>
        private bool WallBackDetected(List<int> collisions)
        {
           // No collisions in the detection list
           if (collisions.Count == 0)
           {
              return false;
           }
           else
           {
              foreach (int sphere in collisions)
              {
                 if (sphere == (int)spheres.BACK)
                 {
                    return true; //  collision Detected
                 }
              }
           }

           return false; // No  collision
        }

        /// <summary>
        /// Detects a collision with the up sphere
        /// </summary>
        /// <param name="collisions"></param>
        /// <returns></returns>
        private bool WallUpDetected(List<int> collisions)
        {
           // No collisions in the detection list
           if (collisions.Count == 0)
           {
              return false;
           }
           else
           {
              foreach (int sphere in collisions)
              {
                 if (sphere == (int)spheres.UP)
                 {
                    return true; //  collision Detected
                 }
              }
           }

           return false; // No collision
        }

        /// <summary>
        /// Detects a collision with the down sphere
        /// </summary>
        /// <param name="collisions"></param>
        /// <returns></returns>
        private bool WallDownDetected(List<int> collisions)
        {
           // No collisions in the detection list
           if (collisions.Count == 0)
           {
              return false;
           }
           else
           {
              foreach (int sphere in collisions)
              {
                 if (sphere == (int)spheres.DOWN)
                 {
                    return true; // Collision Detected
                 }
              }
           }

           return false; // no collision
        }

        /// <summary>
        /// Detects a collision with the body sphere
        /// </summary>
        /// <param name="collisions"></param>
        /// <returns></returns>
        private bool WallBodyDetected(List<int> collisions)
        {
           // No collisions in the detection list
           if (collisions.Count == 0)
           {
              return false;
           }
           else
           {
              foreach (int sphere in collisions)
              {
                 if (sphere == (int)spheres.BODY)
                 {
                    return true; // Collision Detected
                 }
              }
           }

           return false; // No collision
        }
        
        /// <summary>
        /// Ground level front collision.
        /// </summary>
        /// <param name="collisions"></param>
        /// <returns></returns>
        private bool GroundFrontDetected(List<int> collisions)
        {
            // No collisions in the detection list
            if (collisions.Count == 0)
            {
                return false;
            }
            else
            {
                foreach (int sphere in collisions)
                {
                    if (sphere == (int)spheres.GROUNDFRONT)
                    {
                        return true; // Collision Detected
                    }
                }
            }

            return false; // No collision
        }

        /// <summary>
        /// Ground level back collision.
        /// </summary>
        /// <param name="collisions"></param>
        /// <returns></returns>
        private bool GroundBackDetected(List<int> collisions)
        {
            // No collisions in the detection list
            if (collisions.Count == 0)
            {
                return false;
            }
            else
            {
                foreach (int sphere in collisions)
                {
                    if (sphere == (int)spheres.GROUNDBACK)
                    {
                        return true; // Collision Detected
                    }
                }
            }

            return false; // No collision
        }

        /// <summary>
        /// Ground level left collision.
        /// </summary>
        /// <param name="collisions"></param>
        /// <returns></returns>
        private bool GroundLeftDetected(List<int> collisions)
        {
            // No collisions in the detection list
            if (collisions.Count == 0)
            {
                return false;
            }
            else
            {
                foreach (int sphere in collisions)
                {
                    if (sphere == (int)spheres.GROUNDLEFT)
                    {
                        return true; // Collision Detected
                    }
                }
            }

            return false; // No collision
        }

        /// <summary>
        /// Ground level right collision.
        /// </summary>
        /// <param name="collisions"></param>
        /// <returns></returns>
        private bool GroundRightDetected(List<int> collisions)
        {
            // No collisions in the detection list
            if (collisions.Count == 0)
            {
                return false;
            }
            else
            {
                foreach (int sphere in collisions)
                {
                    if (sphere == (int)spheres.GROUNDRIGHT)
                    {
                        return true; // Collision Detected
                    }
                }
            }

            return false; // No collision
        }

        #endregion

        #region Public Methods

        /// <summary>
        /// Update all the actions of the player
        /// including animations, rotation, movement, and firing bullets.
        /// Update the Camera.
        /// </summary>
        /// <param name="gameTime">The current Game Time</param>
        /// <param name="newKeyboardState">The player's new keyboard state</param>
        /// <param name="newMouseState">The player's new mouse state</param>
        public void Update(GameTime gameTime, KeyboardState newKeyboardState, MouseState newMouseState)
        {
            if (_health == 0)
            {
                if (_model.AnimationIndex != DIE)
                {
                    _model.AnimationIndex = DIE;
                    _model.PlaybackStatus = false;
                    _model.Play(gameTime);
                    _model.Update(gameTime);
                }
                else if (gameTime.TotalGameTime.Subtract(_model.LastPlayCommand).TotalMilliseconds < 1300)//_model.Animations[DIE].Duration.TotalMilliseconds)
                {
                    _model.Update(gameTime);
                }
                else
                {
                    _isAlive = false;
                }   
            }
            else
            {
                // Calculate the time difference
                float timeDifference = (float)gameTime.ElapsedGameTime.TotalMilliseconds / 1000.0f;

                // Get the Keyboard Input
                UpdateKeyboard(gameTime, newKeyboardState);
                UpdatePlayerSpheres();
                
                // Update the Model Animation
                _model.Update(gameTime);
                    
                // Get the mouse input and recalculate the world matrix
                bool fireBullet = UpdateMouse(newMouseState, gameTime);
                _model.CalculateWorld();

                // Create a YPR matrix for defining the camera and bullet vectors.
                Matrix camAndBullets = Matrix.CreateFromYawPitchRoll(0.0f, _camera.Pitch, 0.0f) * _model.World;            
                Vector3 upOffset = _model.Up * 22.0f;
                Vector3 forwardOffset = _model.Forward * -20.0f;
                Vector3 temp = _model.World.Left;

                // Set the camera Position Relative to the model
                _camera.Position = _model.Position + upOffset + (forwardOffset * 1);
                _camera.Forward = _camera.Position - camAndBullets.Forward;
                _camera.Up = _model.Up;
                
                // Set the bullet start position and fire the bullet
                if (fireBullet && _weaponState < OVERHEAT)
                {
                    Shoot(_camera.Position, camAndBullets.Forward, _camera.Up);
                    _weaponState++;
                }

                if (gameTime.TotalGameTime.Subtract(_coolDown).TotalMilliseconds > 300 && _weaponState < OVERHEAT)
                {
                    _weaponState--;
                    _coolDown = gameTime.TotalGameTime;
                }
                else if (_weaponState == OVERHEAT && gameTime.TotalGameTime.Subtract(_coolDown).TotalSeconds > OVERHEATWAIT)
                {
                    _weaponState = 5;
                    _coolDown = gameTime.TotalGameTime;
                }
                
                _weaponState = (int)MathHelper.Clamp(_weaponState, 0, OVERHEAT);
                
                // Update bullet positions
                UpdateBullets();

            }
        }

        #endregion
    }
}
