<?php if ( ! defined('BASEPATH')) exit('No direct script access allowed');

require_once(APPPATH . 'classes/LDAPDirectory.php');

class ActiveDirectory extends LDAPDirectory {
	protected $defaultSearchAttributes = array("uaIdentifier", "displayName", 'givenname', 'sn', "name", 'lastLogon');
	protected $allFilter = "(|(name=SearchString)(uaIdentifier=SearchString)(uaIdentifier=xSearchString)(givenname=SearchString)(sn=SearchString))";
	protected $nameFilter = "(|(givenname=SearchString)(sn=SearchString))";
	protected $UAIDFilter = "(|(uaIdentifier=SearchString)(uaIdentifier=xSearchString))";
	protected $usernameFilter = "(name=SearchString)";
	
	const ACCOUNT_ENABLED = 512;
	const ACCOUNT_DISABLED = 514;
	
    function __construct($dbType = null) {
		if(is_null($dbType))
			parent::__construct('ad');
		else parent::__construct($dbType);
    }
	
	public function get($uid, $searchAttributes = null) {
		if(is_null($searchAttributes)) $searchAttributes = $this->defaultSearchAttributes;
		
		if(is_a($uid, "ActiveObject")) {
			if(!is_a($uid, ActiveObject::USER)) throw new ActiveObjectRequiredException('Invalid Active Object for query.');
			$uid = $uid->id;
		}
		
		if(is_numeric($uid)) $filter = "(uaIdentifier=$uid)";
		else $filter = "(name=$uid)";
		
		$attributes = parent::get($searchAttributes, $filter);
		$this->process($attributes);
		
		return $attributes;
	}
	
	
	protected function process(&$attributes) {
		//Process database-specific attributes
		if(isset($attributes['lastLogon'])) {
			foreach($attributes['lastLogon'] as &$attribute) {
				$attribute = date('n/j/Y h:i A', self::parseDate($attribute));
			}
		}
	}
	
	public function search($filter, $search) {
		if(is_numeric($search))			$filter = $this->UAIDFilter;
		elseif($filter == 'all')		$filter = $this->allFilter;
		elseif($filter == 'name')		$filter = $this->nameFilter;
		elseif($filter == 'uaid')		$filter = $this->UAIDFilter;
		elseif($filter == 'username')	$filter = $this->usernameFilter;
		else throw new Exception('Invalid search filter.');
		
		return parent::search(array("uaIdentifier", "displayName", "name"), str_replace('SearchString', $search, $filter));
	}
	
	public function auth($user, $password) {
		$usernames = $this->getAuthUsernames($user);
		
		foreach($usernames as $username) {
			if(parent::auth($username, $password) === true) return true;
		}
		
		return false;
	}
	
	protected function getAuthUsernames($user) {
		$usernames = array();
		array_push($usernames, "cn=". $user .",ou=Admin,dc=uaa,dc=alaska,dc=edu");
		array_push($usernames, "cn=". $user .",ou=Students,dc=uaa,dc=alaska,dc=edu");
	
		return $usernames;
	}
	
	/**
	 * Enables the account according to Microsoft's defined userAccountControl AD attribute
	 * Source: http://support.microsoft.com/default.aspx?scid=kb;en-us;305144
	 **/
	public function enable(ActiveObject $ao) {
		$user = $this->get($ao, array('dn'));
		if($user) $this->modify($user['dn'], array('userAccountControl' => self::ACCOUNT_ENABLED));
		else throw new Exception("User account not found in SunOne.");
	}
	
	/**
	 * Disables the account according to Microsoft's defined userAccountControl AD attribute
	 * Source: http://support.microsoft.com/default.aspx?scid=kb;en-us;305144
	 **/
	public function disable(ActiveObject $ao) {
		$user = $this->get($ao, array('dn'));
		if($user) $this->modify($user['dn'], array('userAccountControl' => self::ACCOUNT_DISABLED));
		else throw new Exception("User account not found in SunOne.");
	}
	
	/**
	 * Determines if the account is enabled according to Microsoft's defined userAccountControl AD attribute
	 * Source: http://support.microsoft.com/default.aspx?scid=kb;en-us;305144
	 **/
	public function isEnabled(ActiveObject $ao) {
		$user = $this->get($ao, array('userAccountControl'));
		if(isset($user['userAccountControl']) && $user['userAccountControl'][0] != self::ACCOUNT_DISABLED) return true;
		else return false;
	}
	
	public function membership($uid) {
		if(is_numeric($uid)) $filter = str_replace('SearchString', $uid, $this->UAIDFilter);
		else $filter = str_replace('SearchString', $uid, $this->usernameFilter);
		
		return parent::membership($filter);
	}
}


?>