<?php

class UserAccount extends ModuleClass {

	/******* Public facing pages *******/
	
	public function viewPasswords() {
		$ao = $this->activeObject(ActiveObject::USER);
		
		$params = $this->checkDefaultPasswords($ao);
		
		foreach(array('sunone', 'federated', 'ad') as $directory) {
			if(!isset($params[$directory])) $params[$directory] = 'Account not found';
			elseif($params[$directory] === false) $params[$directory] = 'Not the default';
		}
		
		$this->render('Passwords.tpl', $params);
	}
	
	public function createIdentity() {
		$this->render('CreateIdentity.tpl');
	
	}
	
	
	/******* Attributes *******/

	public function hasDeliveryOption() {
		$ao = $this->activeObject(ActiveObject::USER);
		$mailDelivery = $this->sunone()->get($ao, array('maildeliveryoption'));
		
		if(isset($mailDelivery['maildeliveryoption']) && is_array($mailDelivery['maildeliveryoption'])) return true;
		else return false;
	}
	
	public function isDefaultPassword() {
		$ao = $this->activeObject(ActiveObject::USER);
	
		$passwords = $this->checkDefaultPasswords($ao);
		
		foreach($passwords as $value)
			if($value === false) return false;
		return true;
	}
	
	/**
	 * Determines if an account is 'normal', ie. it has all of its LDAP accounts enabled.
	 * Note that SunOne does not have User Account Control and so uses the 'activated' attribute
	 * to determine if it is enabled.
	 **/
	public function isNormal() {
		$ao = $this->activeObject(ActiveObject::USER);
		
		try {
			foreach(array('ad', 'federated', 'sunone') as $dir) {
				$$dir = $this->$dir()->isEnabled($ao);
			}
			
			if($ad && $federated && $sunone) return true;
			elseif(!$ad && !$federated && !$sunone) return false;
			else printf("Active Directory: %s<br/>Federated: %s<br/>SunOne: %s", $ad ? "Yes" : "No", $federated ? "Yes" : "No", $sunone ? "Yes" : "No");
			return false;
		}
		catch(Exception $e) {
			return "Error encountered:<br/>" . $e->getMessage();
		}
	}
	
	/******* Buttons *******/
	
	public function addDeliveryOption() {
		$ao = $this->activeObject(ActiveObject::USER);
		$user = $this->sunone()->get($ao, array('maildeliveryoption'));
		if(!isset($user['dn'])) return $this->renderError("Account could not be found.", $ao->url);
		
		$this->sunone()->add($user['dn'], array('maildeliveryoption' => 'forward'));
		redirect($ao->url);
	}
	
	public function removeDeliveryOption() {
		$ao = $this->activeObject(ActiveObject::USER);
		$user = $this->sunone()->get($ao, array('maildeliveryoption'));
		if(!isset($user['dn'])) return $this->renderError("Account could not be found.", $ao->url);
		
		$this->sunone()->remove($user['dn'], array('maildeliveryoption' => $user['maildeliveryoption'][0]));
		redirect($ao->url);
	}
	
	public function resetPassword() {
		$ao = $this->activeObject(ActiveObject::USER);
		
		$newPassword = $this->generateDefaultPassword($ao);
		
		try {
			if(empty($newPassword)) throw new Exception("Unable to generate the default password for '" . $ao->uid . "'.");
			
			$unicodeNewPassword = "\"$newPassword\"";
			$newPassw = "";
			
			$len = strlen($unicodeNewPassword);
			for ($i = 0; $i < $len; $i++){
				$newPassw .= "{$unicodeNewPassword{$i}}\000";}
			$unicodeNewPassword = $newPassw;
			
			$logEntry = "Reset password for <a href='$ao->url'>$ao->uid</a> to $newPassword";
			$resetIssue = false;

			$user = $this->federated()->get($ao, array('dn'));
			if($user) $fedResult = $this->federated()->modify($user['dn'], array('unicodePwd' => $unicodeNewPassword));
			else {
				$logEntry .= "<br/>User account not found in Federated directory.";
				$resetIssue = true;
			}
			
			$user = $this->sunone()->get($ao, array('dn'));
			if($user) $sunResult = $this->sunone()->modify($user['dn'], array('userPassword' => "{sha}".base64_encode(pack("H*", sha1($newPassword)))));
			else {
				$logEntry .= "<br/>User account not found in SunOne directory.";
				$resetIssue = true;
			}
			
			$user = $this->ad()->get($ao, array('dn'));
			if($user) $adResult = $this->ad()->modify($user['dn'], array('unicodePwd' => $unicodeNewPassword));
			else {
				$logEntry .= "<br/>User account not found in Active Directory.";
				$resetIssue = true;
			}
			
			$this->log("Password Reset", $logEntry);
			
			if(!$resetIssue) redirect("/module/load/" . get_class() . "/viewPasswords/");
			else $this->renderError($logEntry, $ao->url);
		}
		catch(Exception $e) {
			$this->error("Password Reset", $e->getMessage());
			$this->renderError("Unable to reset ". $ao->uid . "'s password. See <a href='/logentries/'>log</a> for details.", "/module/load/" . get_class() . "/viewPasswords/", $e);
		}
	}
	
	
	public function createAccount() {
		try {
			$accountType = $_REQUEST['accountType'];
			$username = $_REQUEST['username'];
			$uaid = $_REQUEST['uaid'];
			$firstName = $_REQUEST['firstName'];
			$lastName = $_REQUEST['lastName'];
			
			//Verify account does not already exist
			if($this->sunone()->get($username, array('dn'))) throw new Exception('Account already exists in SunOne');
			if($this->federated()->get($username, array('dn'))) throw new Exception('Account already exists in Federated');
			if($this->ad()->get($username, array('dn'))) throw new Exception('Account already exists in AD');
			
			//Build global attributes
			$attrs = array();
			$attrs['givenName'] = $firstName;
			$attrs['sn'] = $lastName;
			$attrs['mail'] = "$username@uaa.alaska.edu";
			
			/******* SunOne Directory Attributes *******/
			$sunDN = "uid=$username,OU=admin,o=uaa.alaska.edu,o=isp";
			
			$objectClass = array();
			array_push($objectClass, "top");
			array_push($objectClass, "person");
			array_push($objectClass, "organizationalPerson");
			array_push($objectClass, "inetorgperson");
			array_push($objectClass, "uaainetorgperson");
			array_push($objectClass, "inetUser");
			array_push($objectClass, "ipUser");
			array_push($objectClass, "nsManagedPerson");
			array_push($objectClass, "userPresenceProfile");
			array_push($objectClass, "inetMailUser");
			array_push($objectClass, "inetLocalMailRecipient");
			
			$sunAttrs = array_merge($attrs, array(
				'uid' => $username,
				'cn' => $firstName . " " . $lastName,
				'uniqueIdentifier' => $uaid,
				'objectClass' => $objectClass));
			
			
			/******* Federated Directory Attributes *******/
			$fedDN = "CN=$username,Ou=userAccounts,ou=uaa,dc=ua,dc=ad,dc=Alaska,dc=edu";
			
			$objectClass = array();
			array_push($objectClass, "top");
			array_push($objectClass, "person");
			array_push($objectClass, "organizationalPerson");
			array_push($objectClass, "user");
			array_push($objectClass, "inetOrgPerson");
			
			$fedAttrs = array_merge($attrs, array(
				'cn' => $username,
				'name' => $username,
				'uaIdentifier' => $uaid,
				'objectClass' => $objectClass));
			
			/******* Active Directory Attributes *******/
			$adDN = "CN=$username,OU=admin,dc=uaa,dc=Alaska,dc=edu";
			
			$objectClass = array();
			array_push($objectClass, "top");
			array_push($objectClass, "person");
			array_push($objectClass, "organizationalPerson");
			array_push($objectClass, "user");
			
			$adAttrs = array_merge($attrs, array(
				'cn' => $username,
				'uaIdentifier' => $uaid,
				'objectClass' => $objectClass));
			
			//Perform account creation
			if(!$this->sunone()->addRecord($sunDN, $sunAttrs)) throw new Exception("$username - Error creating SunOne account");
			if(!$this->federated()->addRecord($fedDN, $fedAttrs)) throw new Exception("$username - Error creating Federated account");
			if(!$this->ad()->addRecord($adDN, $adAttrs)) throw new Exception("$username - Error creating AD account");
			
			$this->log('Create Identity', $accountType . ": <a href='/user/info/$username/'>$username</a>");
			redirect("/user/info/$username/");
		}
		catch(Exception $e) {
			$this->error('Create Identity', $e->getMessage());
			return $this->renderError(null, "/module/load/" . get_class() . '/createIdentity/', $e);
		}
	}
	
	public function normalize() {
		$ao = $this->activeObject(ActiveObject::USER);
		
		try {
			$logEntry = "<a href='$ao->url'>$ao->uid</a>";
			
			foreach(array('ad', 'federated', 'sunone') as $dir) {
				try {
					$this->$dir()->enable($ao);
				}
				catch(Exception $e) {
					$logEntry .= "<br/>" . $e->getMessage();
				}
			}
			
			$this->log("Normalize Account", $logEntry);
			redirect($ao->url);
		}
		catch(Exception $e) {
			$this->error("Normalize Account", $e->getMessage());
			$this->renderError("Unable to normalize account.", $ao->url, $e);
		}
	}
	
	public function disable() {
		$ao = $this->activeObject(ActiveObject::USER);
		
		try {
			$logEntry = "<a href='$ao->url'>$ao->uid</a>";

			foreach(array('ad', 'federated', 'sunone') as $dir) {
				try {
					$this->$dir()->disable($ao);
				}
				catch(Exception $e) {
					$logEntry .= "<br/>" . $e->getMessage();
				}
			}
			
			$this->log("Disable Account", $logEntry);
			redirect($ao->url);
		}
		catch(Exception $e) {
			$this->error("Disable Account", $e->getMessage());
			$this->renderError("Unable to disable account.", $ao->url, $e);
		}
	}
	
	/******* Supporting methods *******/
	
	/**
	 * Takes a user account and generates all old and new possible default passwords
	 * it may have. Returns all passwords in an array.
	 **/
	private function getPotentialDefaultPasswords(ActiveObject $ao) {
		$user = $this->sunone()->get($ao);
		$fedUser = $this->federated()->get($ao, array('initials'));
		
		$last4 = substr($user['uniqueidentifier'][0], -4);
		
		$initials = $user['givenname'][0];
		if(isset($user['middlename'])) $initials .= $user['middlename'][0];
		$initials .= $user['sn'][0];
		
		
		if(isset($user['middlename'])) $smallInitials = $initials[0] . $initials[2];

		$passwords = array();
		array_push($passwords, 'uaa' . strtoupper($initials) . $last4);
		array_push($passwords, 'uaa' . strtoupper($initials) . "!" . $last4);
		array_push($passwords, $user['uniqueidentifier'][0]);
		if(isset($smallInitials)) {
			array_push($passwords, 'uaa' . strtoupper($smallInitials) . $last4);
			array_push($passwords, 'uaa' . strtoupper($smallInitials) . "!" . $last4);
		}
		
		if(count($fedUser) > 0) {
			$fedInitials = $fedUser['initials'][0];
			array_push($passwords, 'uaa' . strtoupper($fedInitials) . $last4);
			array_push($passwords, 'uaa' . strtoupper($fedInitials) . "!" . $last4);
		}
		
		return $passwords;
	}
	
	
	/**
	 * Generates the default password for the account according to UAA standards.
	 * Format: uaa[Initials][Last 4 of UAID]
	 **/
	private function generateDefaultPassword(ActiveObject $ao) {
		$user = $this->federated()->get($ao, array('initials'));
		
		if($user) {
			$last4 = substr($ao->id, -4);
			$initials = $user['initials'][0];
		}
		else {
			//Generate from SunOne as a fallback
			$user = $this->sunone()->get($ao, array('middlename'));
			
			if(!$user) return false;
			
			$last4 = substr($ao->id, -4);
			
			$initials = $ao->firstName;
			if(isset($user['middlename'])) $initials .= $user['middlename'][0][0];
			$initials .= $ao->lastName;
		}
		
		return 'uaa' . strtoupper($initials) . $last4;
	}
	
	/**
	 * Checks each LDAP directory for a default password by iterating through all possible default
	 * passwords. Returns an associative array in the form ('directory' => 'password').
	 * Example: 'sunone' => 'uaa!1324'.
	 * If the password is not found, the 'password' entry will be false.
	 * If the account is not found in the directory, that entry will be omitted from the array.
	 **/
	private function checkDefaultPasswords(ActiveObject $ao) {
		$passwords = array();
		
		foreach(array('sunone', 'federated', 'ad') as $directory) {
			$pass = $this->getPassword($this->$directory(), $ao);
			if($pass != false) $passwords[$directory] = $pass;
			elseif($this->$directory()->get($ao)) $passwords[$directory] = false;
		}
		
		return $passwords;
	}
	
	/**
	 * Attempts to authenticate all possible default passwords for an account with the
	 * passed LDAPDirectory. Returns the first password that succeeds or false upon failure.
	 **/
	private function getPassword(LDAPDirectory $directory, ActiveObject $ao) {
		$passwords = $this->getPotentialDefaultPasswords($ao);
		
		foreach($passwords as $password) {
			if($directory->auth($ao->uid, $password)) return $password;
		}
		
		return false;
	}
}


?>