/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.me.sealproject.controllers.datacontrollers;

import android.content.ContentValues;
import android.content.Context;
import android.database.Cursor;
import android.database.sqlite.SQLiteDatabase;


import java.util.ArrayList;
import java.util.Observable;
import java.util.Observer;

import org.me.sealproject.controllers.SealController;
import org.me.sealproject.database.*;
import org.me.sealproject.sealdatatypes.*;

/**
 * Controls access to the seal database
 * @author samuelgbeecher
 */

public class SealDatabaseController extends SealController{

    public static final String DATABASE_NAME = "Seal_Database.db";

    private SQLiteDatabase database = null;
    private SealDatabaseHelper helper = null;

    private DatabaseInsertionObserverable observable;

    public SealDatabaseController(Context context){
        super(context);
        helper = new SealDatabaseHelper(context, DATABASE_NAME, null, 1);
        observable = new DatabaseInsertionObserverable();
    }

    public void openOrCreateDatabase(){
        database = helper.getReadableDatabase();
        
        if(database != null)
            LOG(">>>>>> Database Successfully Opened");
    }

    public boolean isOpen(){
        return database.isOpen();
    }

    public void close(){
        if(database != null){
            database.close();
            LOG( "<<<<<< Database Successfully Closed");
        }
        else{
            LOG("Attempting to close an unopened database");
        }
    }

    public void addObserver(Observer obs){
        observable.addObserver(obs);
    }

    /**
     * Insert the given observation into the database. No error checking is performed.
     * @param observation - SealObservation to be inserted. The new id will be assigned to this object.
     * @return true if successful, false otherwise.
     */
    public boolean insertObservation(SealObservation obs){
        ContentValues values = Observations.getValuesFromData(obs);

        long id = database.insert(Observations.NAME, null, values);

        if(id < 0){
            return false;
        }
        else{
            obs.setId(id);
            return true;
        }
    }

    public boolean updateObservation(SealObservation obs){
        ContentValues values = Observations.getValuesFromData(obs);
        values.remove(Observations.ID);
        int rowsChanged = 0;
        
        try{
             rowsChanged = database.update(Observations.NAME, values, "ROWID=?", new String[]{obs.getId()+""});
        }
        catch(Exception e){
            ERROR("Error Updating Observation", e);
        }

        return rowsChanged > 0;
    }

    /**
     * Insert data into the database. Id of given argument will be set after insertion.
     * @param count - SealCount to be inserted
     * @return - true if successful, false otherwise.
     */
    public boolean insertCount(SealCount count){
        ContentValues values =  Counts.getValuesFromData(count);

        long id = database.insert(Counts.NAME, null, values);
        if(id < 0){
            return false;
        }
        else{
            count.setId(id);
            return true;
        }
    }

    public boolean insertTag(SealTag tag){
        ContentValues values = Tags.getValuesFromData(tag);

        if(values == null)
            return false;

        long id = database.insert(Tags.NAME, null, values);
        
        if(id < 0)
            return false;
        else
            return true;
    }

    public boolean updateTag(SealTag tag){
        ContentValues values = Tags.getValuesFromData(tag);

        if(values == null)
            return false;

        String color = tag.getColor();

        if(tag.getColor() == null || tag.getColor().length() < 1)
            color = "NONE";


        int rowsChanged = database.update(Tags.NAME, values, 
                Tags.NUMBER+"=? AND "+Tags.COLOR+"=? AND "+Tags.TYPE+"=?",
                new String[]{ tag.getNumber(), color, tag.getType() });

        return rowsChanged > 0;
    }

    public boolean clearTagRecords(){
        return database.delete(Tags.NAME, null, null) > 0;
    }

    public boolean tagExists(SealTag tag){
        String color = null;

        try{
            if(tag.getColor() != null && tag.getColor().length() >= 1){
                color = tag.getColor();
            }
            else{
                color = "NULL";
            }
        }
        catch(Exception e){
            ERROR("Color not good", e);
            color = "NULL";
        }

        Cursor c  = database.query(Tags.NAME, new String[]{Tags.NUMBER},
                Tags.NUMBER+"=? AND "+Tags.COLOR+"=? AND "+Tags.TYPE+"=?",
                new String[]{tag.getNumber(), color, tag.getType()},
                null, null, null);

        int count = c.getCount();

        c.close();
        
        return count > 0;
    }

    /**
     * Insert the given seal along with its tags into the database.
     * @param seal - Seal to be inserted
     * @return true if seal and its tags were inserted successfully successful, false otherwise.
     */
    public boolean insertSealAndTags(Seal seal){
        ContentValues values = Seals.getValuesFromData(seal);

        long id = database.insert(Seals.NAME, null, values);

        if(id < 0){
            return false;
        }

        SealTag firstTag = seal.getFirstTag();
        SealTag secondTag = seal.getSecondTag();

        if(firstTag != null && !insertTag(firstTag)){
            LOG("First tag failed to insert");
            return false;
        }
        else if(secondTag != null && !insertTag(secondTag)){
            LOG("Second tag failed to insert");
            return false;
        }
        else{
            return true;
        }
    }

    public boolean updateSealAndTags(Seal seal, boolean insertIfNecessary){
        boolean successful;

        successful = updateSeal(seal);

        if(!successful)
            return false;

        if(insertIfNecessary){
            SealTag firstTag = seal.getFirstTag();

            if(firstTag != null && !tagExists(firstTag)){
                successful = insertTag(firstTag);
            }

            if(!successful)
                return false;

            SealTag secondTag = seal.getSecondTag();

            if(secondTag != null && !tagExists(secondTag)){
                successful = insertTag(firstTag);
            }

            if(!successful)
                return false;
        }

        return true;

    }

    public boolean updateSeal(Seal seal){
        ContentValues values = Seals.getValuesFromData(seal);
        
        if(values == null)
            return false;


        int rowsChanged = database.update(Seals.NAME, values, Seals.ID+"=?", new String[]{seal.getId()});

        return rowsChanged > 0;
    }

    public boolean sealExists(Seal seal){
        Cursor c = database.query(Seals.NAME, new String[]{Seals.ID}, Seals.ID+"=?", new String[]{seal.getId()}, null, null, null);

        int count = c.getCount();
        
        c.close();

        return count > 0;
    }

    public boolean insertOrUpdateSealAndTags(Seal seal){
        if(sealExists(seal)){
            return updateSealAndTags(seal, true);
        }
        else{
            return insertSealAndTags(seal);
        }
    }

    public boolean insertMultipleSealAndTags(ArrayList<Seal> seals){
        int index = 0;
        
        DatabaseNotification notification = new DatabaseNotification();

        while(index < seals.size()){
             database.beginTransaction();
             
             try{
                 LOG("Inserting " + index + " - " + (index+300));
                 //Insert in groups to decrease wait time
                 //Downside is any failed insert will result in the entire set not being inserted.
                  for(int i = 0; i < 300 && index < seals.size(); i++){
                        //insertOrUpdateSealAndTags(seals.get(index));
                        insertSealAndTags(seals.get(index));
                        index++;
                  }
                  database.setTransactionSuccessful();

                  observable.notifyObservers(new Integer(index));
                  
             }
             catch(Exception e){
                 ERROR("Database Insertion Error", e);
             }
             finally{
                 database.endTransaction();
             }
        }

        return true;
    }

    public boolean clearSealRecords(){
        return database.delete(Seals.NAME, null, null) > 0;
    }

    /**
     * Return the tag with the given tagNumber. Null is returned if the tag number does not exist.
     */
    public String getSealIdForTag(SealTag tag){
        if(tag.getType() == null)
            return null;

        LOG("Finding Seal for Tag: " + tag);
        
        String color = "NULL";

        if(tag.getColor() != null && tag.getColor().length() > 0){
            color = tag.getColor();
        }
        
        String id = null;

        Cursor c = database.query(Tags.NAME,
                                new String[]{Tags.SEAL_ID},
                                Tags.NUMBER+"=? AND "+Tags.COLOR+"=? AND "+Tags.TYPE+"=?",
                                new String[]{tag.getNumber(), color, tag.getType()},
                                null, null, null);
        
        if(c.moveToFirst()){
            id = c.getString(c.getColumnIndex(Tags.SEAL_ID));
        }

        c.close();

        return id;
    }
    
    /**
     * Get the seal that is linked to the given tag number. Useful for error checking input Tag Numbers.
     * Seal does not have first/second tag populated. To do so, call getTagsForSeal() afterwards. 
     * @param tagNumber - tag number being checked
     * @return Seal if it exists, null otherwise.
     */
    public Seal getSealWithTag(SealTag tag){
        Cursor sealCursor = null;
        Seal seal = null;

        tag.setSealId(getSealIdForTag(tag));


        LOG("Finding Seal for Tag: " + tag);
        if(tag.getSealId() != null){
            LOG("Seal Found for Seal Tag");
            sealCursor = database.query(Seals.NAME,
                    null,
                    Seals.ID+"=?",
                    new String[]{(tag.getSealId())},
                    null, null, null);


            if(sealCursor.moveToFirst()){
                seal = Seals.getDataFromCursor(sealCursor);

                // Get the tags fro the seal and add them to the found seal.
                // This is redundant, since the first tag had already been found. But I see no convenient way to do this otherwise.
                // Could customize a SQLite select statement, but not sure if that would be profitable.
                SealTag[] tags = getTagsForSeal(seal);


                seal.setFirstTag(tags[0]);
                
                if(tags.length > 1)
                    seal.setSecondTag(tags[1]);
            }
        }

        if(sealCursor != null)
            sealCursor.close();

        return seal;
    }

    /**
     * Return all seal tags for the given seal. Search is based on sealId
     * @param seal
     * @return
     */
    public SealTag[] getTagsForSeal(Seal seal){
        SealTag[] tags = null;

        Cursor c = database.query(Tags.NAME,
                null,
                Tags.SEAL_ID+"=?",
                new String[]{seal.getId()},
                null, null, null);

        if(c.moveToFirst()){
            tags = new SealTag[c.getCount()];
            for(int i = 0; i < tags.length; i++){
                tags[i] = Tags.getDataFromCursor(c);
                c.moveToNext();
            }
        }

        c.close();

        return tags;
    }

    /**
     * @return All tag numbers in the database
     */
    public String[] getAllTagNumbers(){
        String[] tagNums = null;
        
        Cursor c = database.query(Tags.NAME,
                new String[]{Tags.NUMBER},
                null,
                null,
                null, null, null);
        
        if(c.moveToFirst()){
            tagNums = new String[c.getCount()];
            for(int i = 0; i < tagNums.length; i++){
                tagNums[i] = c.getString(c.getColumnIndex(Tags.NUMBER));
                c.moveToNext();
            }
        }

        c.close();
        return tagNums;
    }

    /**
     * Get all observations made on the given date.
     * @param unixDateTimeInSeconds Unix time stamp date.
     * @return array of SealObservations for that date
     */
    public SealObservation[] getObservationsOnDate(long unixDateTimeInSeconds){
        SealObservation[] observations = null;

        Cursor cursor = database.query(Observations.NAME,
                    null,
                    "datetime("+Observations.DATETIME+", 'unixepoch', 'localtime', 'start of day') = datetime(?,'unixepoch', 'localtime', 'start of day')",
                    new String[]{unixDateTimeInSeconds+""},
                    null, null, null);


        if(cursor.moveToFirst()){
            observations = new SealObservation[cursor.getCount()];
            for(int i = 0; i < observations.length; i++){
                observations[i] = Observations.getDataFromCursor(cursor);
                cursor.moveToNext();
            }
        }

        cursor.close();

        return observations;
    }

    /**
     * Get all observations after the given Unix Timestamp date
     * @param dateTime - Unix Timestamp date
     * @return Array of all observations after the given date or null if none.
     */
    public SealObservation[] getObservationsAfterDate(long dateTime){

        SealObservation[] observations = null;

        Cursor cursor = database.query(Observations.NAME,
                    null,
                    Observations.DATETIME+" > ?",
                    new String[]{dateTime+""},
                    null, null, null);
        

        if(cursor.moveToFirst()){
            observations = new SealObservation[cursor.getCount()];
            for(int i = 0; i < observations.length; i++){
                observations[i] = Observations.getDataFromCursor(cursor);
                cursor.moveToNext();
            }
        }

        cursor.close();

        return observations;
    }

    public SealCount[] getCountsOnDate(long unixDateTime){
        SealCount[] counts = null;

        Cursor cursor = database.query(Counts.NAME,
                    null,
                    "datetime("+Counts.START_ENTRY_TIME+", 'unixepoch', 'localtime', 'start of day') = datetime(?,'unixepoch', 'localtime', 'start of day')",
                    new String[]{unixDateTime+""},
                    null, null, null);


        if(cursor.moveToFirst()){
            counts = new SealCount[cursor.getCount()];
            for(int i = 0; i < counts.length; i++){
                counts[i] = Counts.getDataFromCursor(cursor);
                cursor.moveToNext();
            }
        }

        cursor.close();

        return counts;
    }

    public class DatabaseInsertionObserverable extends Observable{

        @Override
        public boolean hasChanged(){
            return true;
        }
    }

    public class DatabaseNotification {
        public int recordsInserted;
    }



}
