/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package org.me.sealproject.customwidgets;

import android.content.Context;
import android.content.res.TypedArray;
import android.util.AttributeSet;
import java.util.ArrayList;
import junit.framework.Assert;

import org.me.sealproject.*;

/**
 * This is a convenience class. Handles two arrays of items. In the rare situation in which two sets of distinct data must be combined in a single spinner, this
 * custom widget provides that functionality. Although the case is rare and oftentimes there are more desirable solutions, this provides a workable way to implement.
 * In the xml, the custom attributes are "topEntries:" and "bottomEntries:"
 * @author samuelgbeecher
 */
public class ComboSpinner extends HashSpinner {
    private ArrayList<String> topKeys, topValues, bottomKeys, bottomValues;

     public ComboSpinner(Context context){
        this (context, null);
    }

    public ComboSpinner(Context context, AttributeSet attrs) {
        super (context, attrs);
        TypedArray myArray = context.obtainStyledAttributes(attrs, R.styleable.ComboSpinner);

        Assert.assertEquals("Must have top keys / values and bottom keys / values entries specified", myArray.getIndexCount(), 4);

        setup(myArray);
    }

    public ComboSpinner(Context context, AttributeSet attrs, int defStyle){
        super (context, attrs, defStyle);

        TypedArray myArray = context.obtainStyledAttributes(attrs, R.styleable.ComboSpinner);

        Assert.assertEquals("Must have top and bottom entries specified", myArray.getIndexCount(), 4);

        setup(myArray);
    }

    // Set up the values based on the given typed array
    private void setup(TypedArray typedArray){
        topKeys = new ArrayList<String>();
        topValues = new ArrayList<String>();
        bottomKeys = new ArrayList<String>();
        bottomValues = new ArrayList<String>();

        CharSequence[] topKeyEntries = typedArray.getTextArray(R.styleable.ComboSpinner_topKeyEntries);
        CharSequence[] topValueEntries = typedArray.getTextArray(R.styleable.ComboSpinner_topValueEntries);

        for(int i = 0; i < topKeyEntries.length; i++){
            putTop(topKeyEntries[i].toString(), topValueEntries[i].toString());
        }

        CharSequence[] bottomKeyEntries = typedArray.getTextArray(R.styleable.ComboSpinner_bottomKeyEntries);
        CharSequence[] bottomValueEntries = typedArray.getTextArray(R.styleable.ComboSpinner_bottomValueEntries);

        for(int i = 0; i < bottomKeyEntries.length; i++){
            putBottom(bottomKeyEntries[i].toString(), bottomValueEntries[i].toString());
        }

        super.setSelectionList(combineArrays(topKeys, bottomKeys), combineArrays(topValues, bottomValues));

        super.refreshDisplay();
    }


    /**
     * Put the given value with the given key. They key is added if it does not exist. Old values are overwritten.
     * @param key
     * @param value
     */
    public void putTop(String key, String value){
        int index = topKeys.indexOf(key);

        if(index < 0){
            topKeys.add(key);
            topValues.add(value);
        }
        else{
            topValues.set(index, value);
        }
    }


    /**
     * Put the given value with the given key. They key is added if it does not exist. Old values are overwritten.
     * @param key
     * @param value
     */
    public void putBottom(String key, String value){
        int index = bottomKeys.indexOf(key);

        if(index < 0){
            bottomKeys.add(key);
            bottomValues.add(value);
        }
        else{
            bottomValues.set(index, value);
        }
    }

    /**
     * Is the given item contained in the top entries
     * @param item - item to be checked against top entries
     * @return true if item exists in the top entries
     */
    public boolean topEntriesContainValue(String value){
        return topValues.contains(value);
    }

    /**
     * Return true if the top entries contains the input key
     * @param key
     * @return
     */
    public boolean topEntriesContainKey(String key){
        return topKeys.contains(key);
    }

    /**
     * Is the given item contained in the bottom entries
     * @param item - item to be checked against bottom entries
     * @return true if item exists in the bottom entries
     */
    public boolean bottomEntriesContainValue(String value){
        return bottomValues.contains(value);
    }

    /**
     * Return true if the bottom entries contain the key
     * @param key
     * @return
     */
    public boolean bottomEntriesContainKey(String key){
        return bottomKeys.contains(key);
    }

    /**
     * Sets the selection to the index as per topEntries.
     * Ex. if you wanted to set the selection to the first bottom entry, you would pass in 0, etc...
     * @param selectionIndex
     */
    public void setSelectionBasedOnTopEntries(int selectionIndex){
        this.setSelectionIgnoringDefault(selectionIndex);
    }

    /**
     * Return the top entry selection index. -1 if top entry is not selected.
     */
    public int getTopEntriesSelection(){
        if(isTopEntrySelected()){
            return this.getSelectedItemPositionIgnoringDefault();
        }
        else
            return -1;
    }

    /**
     * Sets the selection to the index as per bottomEntries.
     * Ex. if you wanted to set the selection to the first bottom entry, you would pass in 0, etc...
     * @param selectionIndex
     */
    public void setSelectionBasedOnBottomEntries(int selectionIndex){
        this.setSelectionIgnoringDefault(selectionIndex+topKeys.size());
    }

    /**
     * Return the index of the bottom entry selection
     * @return
     */
    public int getBottomEntriesSelection(){
        if(isBottomEntrySelected()){
            return this.getSelectedItemPositionIgnoringDefault() - topKeys.size();
        }
        else
            return -1;
    }

    /**
     * Check whether a top entry has been selected
     * @return True if a top entry has been selected, false if either no selection has been made (default text is selected) or bottom entry is selected
     */
    public boolean isTopEntrySelected(){
        if(this.getSelectedItemPositionIgnoringDefault() < 0){
            return false; //No entry has been selected
        }
        else{
            return this.getSelectedItemPositionIgnoringDefault() < topKeys.size();
        }
    }

    /**
     * Check whether the bottom entry has been selected
     * @return true if a bottom entry has been selected. False if not. 
     */
    public boolean isBottomEntrySelected(){
        return this.getSelectedItemPositionIgnoringDefault() >= topKeys.size();
    }

    public void setTopSelectionList(ArrayList<String> keys, ArrayList<String> values){
        topKeys = keys;
        topValues = values;

        super.setSelectionList(combineArrays(topKeys, bottomKeys), combineArrays(topValues, bottomValues));
    }

    public void setBottomSelectionList(ArrayList<String> keys, ArrayList<String> values){
        bottomKeys = keys;
        bottomValues = values;
        
        super.setSelectionList(combineArrays(topKeys, bottomKeys), combineArrays(topValues, bottomValues));
    }

    /**
     * This is a convenience method for combining my internal arrays into an appropriate array list.
     * This is necessary for posting the new data to the Spinner.
     * @param top
     * @param bottom
     * @return
     */
    private ArrayList<String> combineArrays(ArrayList<String> top, ArrayList<String> bottom){
        int index = 0;
       ArrayList<String> combined = new ArrayList<String>(top.size() + bottom.size());

        for(int i = 0; i < top.size(); i++, index++){
            combined.add(top.get(i));
        }

        for(int i = 0; i < bottom.size(); i++, index++){
            combined.add(bottom.get(i));
        }

        return combined;
    }
}
